import { mysqlTable, text, int, boolean, timestamp, varchar } from "drizzle-orm/mysql-core";
import { createInsertSchema } from "drizzle-zod";
import { z } from "zod";

export const users = mysqlTable("users", {
  id: int("id").primaryKey().autoincrement(),
  username: varchar("username", { length: 255 }).notNull().unique(),
  password: text("password").notNull(),
});

export const insertUserSchema = createInsertSchema(users).pick({
  username: true,
  password: true,
});

export type InsertUser = z.infer<typeof insertUserSchema>;
export type User = typeof users.$inferSelect;

// Categories for restaurant menu
export const categories = mysqlTable("categories", {
  id: int("id").primaryKey().autoincrement(),
  name: varchar("name", { length: 255 }).notNull(),
  slug: varchar("slug", { length: 255 }).notNull().unique(),
  order: int("order").notNull(),
});

export const insertCategorySchema = createInsertSchema(categories).pick({
  name: true,
  slug: true,
  order: true,
});

export type InsertCategory = z.infer<typeof insertCategorySchema>;
export type Category = typeof categories.$inferSelect;

// Menu items for restaurant
export const menuItems = mysqlTable("menu_items", {
  id: int("id").primaryKey().autoincrement(),
  name: varchar("name", { length: 255 }).notNull(),
  description: text("description"),
  price: varchar("price", { length: 20 }).notNull(),
  categoryId: int("category_id").notNull(),
  image: text("image"),
  images: text("images"),
  isActive: boolean("is_active").default(true),
  isVegetarian: boolean("is_vegetarian").default(false),
  isVegan: boolean("is_vegan").default(false),
  isGlutenFree: boolean("is_gluten_free").default(false),
  isSpicy: boolean("is_spicy").default(false),
  isChefSpecial: boolean("is_chef_special").default(false),
  isMonthlySpecial: boolean("is_monthly_special").default(false),
  isCampaign: boolean("is_campaign").default(false),
  campaignPrice: varchar("campaign_price", { length: 20 }),
  sortOrder: int("sort_order").default(0),
  preparationTime: varchar("preparation_time", { length: 50 }),
  allergens: text("allergens"),
  createdAt: timestamp("created_at").defaultNow(),
});

export const insertMenuItemSchema = createInsertSchema(menuItems).pick({
  name: true,
  description: true,
  price: true,
  categoryId: true,
  image: true,
  images: true,
  isActive: true,
  isVegetarian: true,
  isVegan: true,
  isGlutenFree: true,
  isSpicy: true,
  isChefSpecial: true,
  isMonthlySpecial: true,
  isCampaign: true,
  campaignPrice: true,
  sortOrder: true,
  preparationTime: true,
  allergens: true,
});

export type InsertMenuItem = z.infer<typeof insertMenuItemSchema>;
export type MenuItem = typeof menuItems.$inferSelect;

// Restaurant information
export const restaurants = mysqlTable("restaurants", {
  id: int("id").primaryKey().autoincrement(),
  name: varchar("name", { length: 255 }).notNull(),
  description: text("description"),
  address: text("address"),
  phone: varchar("phone", { length: 20 }),
  email: varchar("email", { length: 255 }),
  website: varchar("website", { length: 255 }),
  logo: text("logo"),
  isActive: boolean("is_active").default(true),
  // Payment system settings
  enablePayments: boolean("enable_payments").default(false),
  enablePaytr: boolean("enable_paytr").default(false),
  enableIyzico: boolean("enable_iyzico").default(false),
  enableCashPayment: boolean("enable_cash_payment").default(true),
  // Commission rates
  paytrCommissionRate: varchar("paytr_commission_rate", { length: 10 }).default("2.9"), // %2.9
  iyzicoCommissionRate: varchar("iyzico_commission_rate", { length: 10 }).default("2.7"), // %2.7
  fixedFeePaytr: varchar("fixed_fee_paytr", { length: 10 }).default("0.25"), // 0.25 TL
  fixedFeeIyzico: varchar("fixed_fee_iyzico", { length: 10 }).default("0.30"), // 0.30 TL
  createdAt: timestamp("created_at").defaultNow(),
});

export const insertRestaurantSchema = createInsertSchema(restaurants).pick({
  name: true,
  description: true,
  address: true,
  phone: true,
  email: true,
  website: true,
  logo: true,
  isActive: true,
  enablePayments: true,
  enablePaytr: true,
  enableIyzico: true,
  enableCashPayment: true,
  paytrCommissionRate: true,
  iyzicoCommissionRate: true,
  fixedFeePaytr: true,
  fixedFeeIyzico: true,
});

export type InsertRestaurant = z.infer<typeof insertRestaurantSchema>;
export type Restaurant = typeof restaurants.$inferSelect;

// Theme settings for customization
export const themeSettings = pgTable("theme_settings", {
  id: serial("id").primaryKey(),
  name: text("name").notNull(),
  primaryColor: text("primary_color").notNull(),
  secondaryColor: text("secondary_color").notNull(),
  backgroundColor: text("background_color").notNull(),
  textColor: text("text_color").notNull(),
  fontFamily: text("font_family").notNull(),
  isActive: boolean("is_active").default(false),
  createdAt: timestamp("created_at").defaultNow(),
});

export const insertThemeSettingsSchema = createInsertSchema(themeSettings).pick({
  name: true,
  primaryColor: true,
  secondaryColor: true,
  backgroundColor: true,
  textColor: true,
  fontFamily: true,
  isActive: true,
});

export type InsertThemeSettings = z.infer<typeof insertThemeSettingsSchema>;
export type ThemeSettings = typeof themeSettings.$inferSelect;

// Images for menu items
export const images = pgTable("images", {
  id: serial("id").primaryKey(),
  url: text("url").notNull(),
  alt: text("alt"),
  title: text("title"),
  size: integer("size"),
  mimeType: text("mime_type"),
  createdAt: timestamp("created_at").defaultNow(),
});

export const insertImageSchema = createInsertSchema(images).pick({
  url: true,
  alt: true,
  title: true,
  size: true,
  mimeType: true,
});

export type InsertImage = z.infer<typeof insertImageSchema>;
export type Image = typeof images.$inferSelect;

// Junction table for menu item images
export const menuItemImages = pgTable("menu_item_images", {
  id: serial("id").primaryKey(),
  menuItemId: integer("menu_item_id").notNull(),
  imageId: integer("image_id").notNull(),
  order: integer("order").default(0),
});

export const insertMenuItemImageSchema = createInsertSchema(menuItemImages).pick({
  menuItemId: true,
  imageId: true,
  order: true,
});

export type InsertMenuItemImage = z.infer<typeof insertMenuItemImageSchema>;
export type MenuItemImage = typeof menuItemImages.$inferSelect;

// SEO settings
export const seoSettings = pgTable("seo_settings", {
  id: serial("id").primaryKey(),
  pageTitle: text("page_title"),
  metaDescription: text("meta_description"),
  metaKeywords: text("meta_keywords"),
  ogTitle: text("og_title"),
  ogDescription: text("og_description"),
  ogImage: text("og_image"),
  twitterTitle: text("twitter_title"),
  twitterDescription: text("twitter_description"),
  twitterImage: text("twitter_image"),
  canonicalUrl: text("canonical_url"),
  robotsContent: text("robots_content"),
  structuredData: text("structured_data"),
  isActive: boolean("is_active").default(true),
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
});

export const insertSeoSettingsSchema = createInsertSchema(seoSettings).pick({
  pageTitle: true,
  metaDescription: true,
  metaKeywords: true,
  ogTitle: true,
  ogDescription: true,
  ogImage: true,
  twitterTitle: true,
  twitterDescription: true,
  twitterImage: true,
  canonicalUrl: true,
  robotsContent: true,
  structuredData: true,
  isActive: true,
});

export type InsertSeoSettings = z.infer<typeof insertSeoSettingsSchema>;
export type SeoSettings = typeof seoSettings.$inferSelect;

// Orders system
export const orders = pgTable("orders", {
  id: serial("id").primaryKey(),
  orderNumber: text("order_number").notNull().unique(),
  tableNumber: text("table_number"),
  customerName: text("customer_name"),
  customerPhone: text("customer_phone"),
  status: text("status").notNull().default("pending"), // pending, confirmed, preparing, ready, delivered, cancelled
  totalAmount: text("total_amount").notNull(),
  notes: text("notes"),
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
});

export const insertOrderSchema = createInsertSchema(orders).pick({
  tableNumber: true,
  customerName: true,
  customerPhone: true,
  totalAmount: true,
  notes: true,
});

export type InsertOrder = z.infer<typeof insertOrderSchema>;
export type Order = typeof orders.$inferSelect;

// Order items
export const orderItems = pgTable("order_items", {
  id: serial("id").primaryKey(),
  orderId: integer("order_id").notNull(),
  menuItemId: integer("menu_item_id").notNull(),
  quantity: integer("quantity").notNull(),
  price: text("price").notNull(),
  notes: text("notes"),
});

export const insertOrderItemSchema = createInsertSchema(orderItems).pick({
  orderId: true,
  menuItemId: true,
  quantity: true,
  price: true,
  notes: true,
});

export type InsertOrderItem = z.infer<typeof insertOrderItemSchema>;
export type OrderItem = typeof orderItems.$inferSelect;

// Customer loyalty system
export const customers = pgTable("customers", {
  id: serial("id").primaryKey(),
  name: text("name").notNull(),
  phone: text("phone").unique(),
  email: text("email").unique(),
  address: text("address"),
  points: integer("points").default(0),
  totalSpent: text("total_spent").default("0"),
  totalOrders: integer("total_orders").default(0),
  visits: integer("visits").default(0),
  lastVisit: timestamp("last_visit"),
  createdAt: timestamp("created_at").defaultNow(),
});

export const insertCustomerSchema = createInsertSchema(customers).pick({
  name: true,
  phone: true,
  email: true,
  address: true,
});

export type InsertCustomer = z.infer<typeof insertCustomerSchema>;
export type Customer = typeof customers.$inferSelect;

// Coupons and promotions
export const coupons = pgTable("coupons", {
  id: serial("id").primaryKey(),
  code: text("code").notNull().unique(),
  name: text("name").notNull(),
  description: text("description"),
  discountType: text("discount_type").notNull(), // percentage, fixed
  discountValue: text("discount_value").notNull(),
  minOrderAmount: text("min_order_amount").default("0"),
  maxUses: integer("max_uses").default(1),
  usedCount: integer("used_count").default(0),
  isActive: boolean("is_active").default(true),
  expiresAt: timestamp("expires_at"),
  createdAt: timestamp("created_at").defaultNow(),
});

export const insertCouponSchema = createInsertSchema(coupons).pick({
  code: true,
  name: true,
  description: true,
  discountType: true,
  discountValue: true,
  minOrderAmount: true,
  maxUses: true,
  isActive: true,
  expiresAt: true,
});

export type InsertCoupon = z.infer<typeof insertCouponSchema>;
export type Coupon = typeof coupons.$inferSelect;

// Analytics tracking
export const analytics = pgTable("analytics", {
  id: serial("id").primaryKey(),
  event: text("event").notNull(), // page_view, menu_item_view, qr_scan, order_placed, etc.
  data: text("data"), // JSON string for additional data
  ipAddress: text("ip_address"),
  userAgent: text("user_agent"),
  timestamp: timestamp("timestamp").defaultNow(),
});

export const insertAnalyticsSchema = createInsertSchema(analytics).pick({
  event: true,
  data: true,
  ipAddress: true,
  userAgent: true,
});

export type InsertAnalytics = z.infer<typeof insertAnalyticsSchema>;
export type Analytics = typeof analytics.$inferSelect;

// Notifications
export const notifications = pgTable("notifications", {
  id: serial("id").primaryKey(),
  title: text("title").notNull(),
  message: text("message").notNull(),
  type: text("type").notNull(), // order, promotion, system
  isRead: boolean("is_read").default(false),
  targetAudience: text("target_audience").default("admin"), // admin, customer, all
  createdAt: timestamp("created_at").defaultNow(),
});

export const insertNotificationSchema = createInsertSchema(notifications).pick({
  title: true,
  message: true,
  type: true,
  targetAudience: true,
});

export type InsertNotification = z.infer<typeof insertNotificationSchema>;
export type Notification = typeof notifications.$inferSelect;

// Reviews and ratings
export const reviews = pgTable("reviews", {
  id: serial("id").primaryKey(),
  orderId: integer("order_id"),
  customerName: text("customer_name").notNull(),
  customerEmail: text("customer_email"),
  rating: integer("rating").notNull(), // 1-5 stars
  comment: text("comment"),
  menuItemId: integer("menu_item_id"), // optional, can be general restaurant review
  isApproved: boolean("is_approved").default(false),
  isPublic: boolean("is_public").default(true),
  createdAt: timestamp("created_at").defaultNow(),
});

export const insertReviewSchema = createInsertSchema(reviews).pick({
  orderId: true,
  customerName: true,
  customerEmail: true,
  rating: true,
  comment: true,
  menuItemId: true,
});

export type InsertReview = z.infer<typeof insertReviewSchema>;
export type Review = typeof reviews.$inferSelect;

// Payment transactions
export const payments = pgTable("payments", {
  id: serial("id").primaryKey(),
  orderId: integer("order_id").notNull(),
  paymentMethod: text("payment_method").notNull(), // paytr, iyzico, cash
  paymentProvider: text("payment_provider"), // paytr, iyzico
  amount: text("amount").notNull(),
  currency: text("currency").default("TRY"),
  status: text("status").notNull().default("pending"), // pending, processing, completed, failed, cancelled, refunded
  transactionId: text("transaction_id"), // External payment provider transaction ID
  paymentToken: text("payment_token"), // Payment provider token
  paymentUrl: text("payment_url"), // Payment redirect URL
  errorMessage: text("error_message"),
  metadata: text("metadata"), // JSON string for additional payment data
  paidAt: timestamp("paid_at"),
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
});

export const insertPaymentSchema = createInsertSchema(payments).pick({
  orderId: true,
  paymentMethod: true,
  paymentProvider: true,
  amount: true,
  currency: true,
});

export type InsertPayment = z.infer<typeof insertPaymentSchema>;
export type Payment = typeof payments.$inferSelect;

// Payment webhooks log
export const paymentWebhooks = pgTable("payment_webhooks", {
  id: serial("id").primaryKey(),
  paymentId: integer("payment_id"),
  provider: text("provider").notNull(), // paytr, iyzico
  webhookType: text("webhook_type").notNull(), // success, failure, etc.
  payload: text("payload").notNull(), // JSON string of webhook data
  processed: boolean("processed").default(false),
  createdAt: timestamp("created_at").defaultNow(),
});

export const insertPaymentWebhookSchema = createInsertSchema(paymentWebhooks).pick({
  paymentId: true,
  provider: true,
  webhookType: true,
  payload: true,
});

export type InsertPaymentWebhook = z.infer<typeof insertPaymentWebhookSchema>;
export type PaymentWebhook = typeof paymentWebhooks.$inferSelect;