import type { Express } from "express";
import { createServer, type Server } from "http";
// PostgreSQL storage (development)
// import { storage } from "./database-storage";

// MySQL storage (production - aktif)
import { storage } from "./database-storage-mysql";
import { z } from "zod";
import { setupAuth } from "./auth";
import { PaymentService } from "./payment-services";
// PostgreSQL schema (development - aktif)
import { insertOrderSchema, insertPaymentSchema } from "@shared/schema";

// MySQL schema (production)
// import { insertOrderSchema, insertPaymentSchema } from "@shared/schema-mysql";

export async function registerRoutes(app: Express): Promise<Server> {
  // Setup authentication routes
  setupAuth(app);
  
  // API routes for restaurant menu system
  
  // Tema ayarları endpointleri
  app.get("/api/themes", async (req, res) => {
    try {
      const theme = await storage.getThemeSettings();
      // API tüm temaları array olarak döndürmeli
      return res.json(theme ? [theme] : []);
    } catch (error) {
      console.error("Error fetching theme settings:", error);
      return res.status(500).json({ message: "Failed to fetch theme settings" });
    }
  });

  app.get("/api/themes/active", async (req, res) => {
    try {
      const theme = await storage.getActiveTheme();
      if (!theme) {
        return res.status(404).json({ message: "No active theme found" });
      }
      return res.json(theme);
    } catch (error) {
      console.error("Error fetching active theme:", error);
      return res.status(500).json({ message: "Failed to fetch active theme" });
    }
  });

  app.post("/api/themes", async (req, res) => {
    try {
      if (!req.isAuthenticated()) {
        return res.status(401).json({ message: "Unauthorized" });
      }

      const theme = await storage.createThemeSettings(req.body);
      return res.status(201).json(theme);
    } catch (error) {
      console.error("Error creating theme:", error);
      return res.status(500).json({ message: "Failed to create theme" });
    }
  });

  app.put("/api/themes/:id", async (req, res) => {
    try {
      if (!req.isAuthenticated()) {
        return res.status(401).json({ message: "Unauthorized" });
      }

      const themeId = parseInt(req.params.id);
      const theme = await storage.updateThemeSettings(themeId, req.body);
      return res.json(theme);
    } catch (error: any) {
      if (error.message && error.message.includes("not found")) {
        return res.status(404).json({ message: error.message });
      }
      console.error("Error updating theme:", error);
      return res.status(500).json({ message: "Failed to update theme" });
    }
  });

  app.put("/api/themes/:id/activate", async (req, res) => {
    try {
      if (!req.isAuthenticated()) {
        return res.status(401).json({ message: "Unauthorized" });
      }

      const themeId = parseInt(req.params.id);
      const theme = await storage.activateTheme(themeId);
      return res.json(theme);
    } catch (error: any) {
      if (error.message && error.message.includes("not found")) {
        return res.status(404).json({ message: error.message });
      }
      console.error("Error activating theme:", error);
      return res.status(500).json({ message: "Failed to activate theme" });
    }
  });

  app.delete("/api/themes/:id", async (req, res) => {
    try {
      if (!req.isAuthenticated()) {
        return res.status(401).json({ message: "Unauthorized" });
      }

      const themeId = parseInt(req.params.id);
      const success = await storage.deleteThemeSettings(themeId);
      if (!success) {
        return res.status(404).json({ message: "Theme not found" });
      }
      return res.status(204).end();
    } catch (error: any) {
      if (error.message && error.message.includes("Cannot delete active theme")) {
        return res.status(400).json({ message: error.message });
      }
      console.error("Error deleting theme:", error);
      return res.status(500).json({ message: "Failed to delete theme" });
    }
  });
  
  // Görsel yönetimi endpointleri
  app.get("/api/images", async (req, res) => {
    try {
      const images = await storage.getImages();
      return res.json(images || []);
    } catch (error) {
      console.error("Error fetching images:", error);
      return res.status(500).json({ message: "Failed to fetch images" });
    }
  });

  app.get("/api/images/:id", async (req, res) => {
    try {
      const imageId = parseInt(req.params.id);
      const image = await storage.getImage(imageId);
      if (!image) {
        return res.status(404).json({ message: "Image not found" });
      }
      return res.json(image);
    } catch (error) {
      console.error("Error fetching image:", error);
      return res.status(500).json({ message: "Failed to fetch image" });
    }
  });

  app.post("/api/images", async (req, res) => {
    try {
      if (!req.isAuthenticated()) {
        return res.status(401).json({ message: "Unauthorized" });
      }

      const image = await storage.createImage(req.body);
      return res.status(201).json(image);
    } catch (error) {
      console.error("Error creating image:", error);
      return res.status(500).json({ message: "Failed to create image" });
    }
  });

  app.put("/api/images/:id", async (req, res) => {
    try {
      if (!req.isAuthenticated()) {
        return res.status(401).json({ message: "Unauthorized" });
      }

      const imageId = parseInt(req.params.id);
      const image = await storage.updateImage(imageId, req.body);
      return res.json(image);
    } catch (error: any) {
      if (error.message && error.message.includes("not found")) {
        return res.status(404).json({ message: error.message });
      }
      console.error("Error updating image:", error);
      return res.status(500).json({ message: "Failed to update image" });
    }
  });

  app.delete("/api/images/:id", async (req, res) => {
    try {
      if (!req.isAuthenticated()) {
        return res.status(401).json({ message: "Unauthorized" });
      }

      const imageId = parseInt(req.params.id);
      const success = await storage.deleteImage(imageId);
      if (!success) {
        return res.status(404).json({ message: "Image not found" });
      }
      return res.status(204).end();
    } catch (error) {
      console.error("Error deleting image:", error);
      return res.status(500).json({ message: "Failed to delete image" });
    }
  });

  // Menü öğesi görselleri endpointleri
  app.get("/api/menu-items/:id/images", async (req, res) => {
    try {
      const menuItemId = parseInt(req.params.id);
      const images = await storage.getMenuItemImages(menuItemId);
      return res.json(images);
    } catch (error) {
      console.error("Error fetching menu item images:", error);
      return res.status(500).json({ message: "Failed to fetch menu item images" });
    }
  });

  app.post("/api/menu-items/:menuItemId/images/:imageId", async (req, res) => {
    try {
      if (!req.isAuthenticated()) {
        return res.status(401).json({ message: "Unauthorized" });
      }

      const menuItemId = parseInt(req.params.menuItemId);
      const imageId = parseInt(req.params.imageId);
      const order = req.body.order ? parseInt(req.body.order) : 0;
      
      const relation = await storage.addImageToMenuItem(menuItemId, imageId, order);
      return res.status(201).json(relation);
    } catch (error: any) {
      if (error.message && error.message.includes("not found")) {
        return res.status(404).json({ message: error.message });
      }
      console.error("Error adding image to menu item:", error);
      return res.status(500).json({ message: "Failed to add image to menu item" });
    }
  });

  app.delete("/api/menu-items/:menuItemId/images/:imageId", async (req, res) => {
    try {
      if (!req.isAuthenticated()) {
        return res.status(401).json({ message: "Unauthorized" });
      }

      const menuItemId = parseInt(req.params.menuItemId);
      const imageId = parseInt(req.params.imageId);
      
      const success = await storage.removeImageFromMenuItem(menuItemId, imageId);
      if (!success) {
        return res.status(404).json({ message: "Relation not found" });
      }
      return res.status(204).end();
    } catch (error) {
      console.error("Error removing image from menu item:", error);
      return res.status(500).json({ message: "Failed to remove image from menu item" });
    }
  });
  
  // Get restaurant information
  app.get("/api/restaurant", async (req, res) => {
    try {
      const restaurantInfo = await storage.getRestaurantInfo();
      
      if (!restaurantInfo) {
        return res.status(404).json({ message: "Restaurant information not found" });
      }
      
      return res.json(restaurantInfo);
    } catch (error) {
      console.error("Error fetching restaurant information:", error);
      return res.status(500).json({ message: "Failed to fetch restaurant information" });
    }
  });
  
  // Create restaurant information
  app.post("/api/restaurant", async (req, res) => {
    try {
      const restaurantInfo = await storage.createRestaurantInfo(req.body);
      return res.status(201).json(restaurantInfo);
    } catch (error) {
      console.error("Error creating restaurant information:", error);
      return res.status(500).json({ message: "Failed to create restaurant information" });
    }
  });
  
  // Update restaurant information
  app.patch("/api/restaurant/:id", async (req, res) => {
    try {
      const id = parseInt(req.params.id);
      
      if (isNaN(id)) {
        return res.status(400).json({ message: "Invalid restaurant ID" });
      }
      
      const updatedInfo = await storage.updateRestaurantInfo(id, req.body);
      return res.json(updatedInfo);
    } catch (error) {
      console.error("Error updating restaurant information:", error);
      return res.status(500).json({ message: "Failed to update restaurant information" });
    }
  });
  
  // Get all categories
  app.get("/api/categories", async (req, res) => {
    try {
      const categories = await storage.getCategories();
      return res.json(categories);
    } catch (error) {
      console.error("Error fetching categories:", error);
      return res.status(500).json({ message: "Failed to fetch categories" });
    }
  });
  
  // Create a new category
  app.post("/api/categories", async (req, res) => {
    try {
      const category = await storage.createCategory(req.body);
      return res.status(201).json(category);
    } catch (error) {
      console.error("Error creating category:", error);
      return res.status(500).json({ message: "Failed to create category" });
    }
  });
  
  // Get a category by slug
  app.get("/api/categories/:slug", async (req, res) => {
    try {
      const { slug } = req.params;
      const category = await storage.getCategoryBySlug(slug);
      
      if (!category) {
        return res.status(404).json({ message: "Category not found" });
      }
      
      return res.json(category);
    } catch (error) {
      console.error("Error fetching category:", error);
      return res.status(500).json({ message: "Failed to fetch category" });
    }
  });
  
  // Update a category
  app.patch("/api/categories/:id", async (req, res) => {
    try {
      const id = parseInt(req.params.id);
      
      if (isNaN(id)) {
        return res.status(400).json({ message: "Invalid category ID" });
      }
      
      const updatedCategory = await storage.updateCategory(id, req.body);
      return res.json(updatedCategory);
    } catch (error) {
      console.error("Error updating category:", error);
      return res.status(500).json({ message: "Failed to update category" });
    }
  });
  
  // Delete a category
  app.delete("/api/categories/:id", async (req, res) => {
    try {
      const id = parseInt(req.params.id);
      
      if (isNaN(id)) {
        return res.status(400).json({ message: "Invalid category ID" });
      }
      
      const result = await storage.deleteCategory(id);
      
      if (!result) {
        return res.status(404).json({ message: "Category not found" });
      }
      
      return res.status(204).end();
    } catch (error) {
      console.error("Error deleting category:", error);
      return res.status(500).json({ message: "Failed to delete category" });
    }
  });
  
  // Get all menu items
  app.get("/api/menu-items", async (req, res) => {
    try {
      const menuItems = await storage.getMenuItems();
      return res.json(menuItems);
    } catch (error) {
      console.error("Error fetching menu items:", error);
      return res.status(500).json({ message: "Failed to fetch menu items" });
    }
  });
  
  // Create a new menu item
  app.post("/api/menu-items", async (req, res) => {
    try {
      const menuItem = await storage.createMenuItem(req.body);
      return res.status(201).json(menuItem);
    } catch (error) {
      console.error("Error creating menu item:", error);
      return res.status(500).json({ message: "Failed to create menu item" });
    }
  });
  
  // Get a menu item by ID
  app.get("/api/menu-items/:id", async (req, res) => {
    try {
      const id = parseInt(req.params.id);
      
      if (isNaN(id)) {
        return res.status(400).json({ message: "Invalid menu item ID" });
      }
      
      const menuItem = await storage.getMenuItem(id);
      
      if (!menuItem) {
        return res.status(404).json({ message: "Menu item not found" });
      }
      
      return res.json(menuItem);
    } catch (error) {
      console.error("Error fetching menu item:", error);
      return res.status(500).json({ message: "Failed to fetch menu item" });
    }
  });
  
  // Update a menu item
  app.patch("/api/menu-items/:id", async (req, res) => {
    try {
      const id = parseInt(req.params.id);
      
      if (isNaN(id)) {
        return res.status(400).json({ message: "Invalid menu item ID" });
      }
      
      const updatedMenuItem = await storage.updateMenuItem(id, req.body);
      return res.json(updatedMenuItem);
    } catch (error) {
      console.error("Error updating menu item:", error);
      return res.status(500).json({ message: "Failed to update menu item" });
    }
  });
  
  // Delete a menu item
  app.delete("/api/menu-items/:id", async (req, res) => {
    try {
      const id = parseInt(req.params.id);
      
      if (isNaN(id)) {
        return res.status(400).json({ message: "Invalid menu item ID" });
      }
      
      const result = await storage.deleteMenuItem(id);
      
      if (!result) {
        return res.status(404).json({ message: "Menu item not found" });
      }
      
      return res.status(204).end();
    } catch (error) {
      console.error("Error deleting menu item:", error);
      return res.status(500).json({ message: "Failed to delete menu item" });
    }
  });
  
  // Get menu items by category ID
  app.get("/api/categories/:id/menu-items", async (req, res) => {
    try {
      const id = parseInt(req.params.id);
      
      if (isNaN(id)) {
        return res.status(400).json({ message: "Invalid category ID" });
      }
      
      const menuItems = await storage.getMenuItemsByCategory(id);
      return res.json(menuItems);
    } catch (error) {
      console.error("Error fetching menu items by category:", error);
      return res.status(500).json({ message: "Failed to fetch menu items" });
    }
  });
  
  // Get complete menu (categories with their items)
  app.get("/api/menu", async (req, res) => {
    try {
      const categories = await storage.getCategories();
      const menuItems = await storage.getMenuItems();
      
      const menu = await Promise.all(
        categories.map(async (category) => {
          const items = menuItems.filter(item => item.categoryId === category.id);
          return {
            ...category,
            items
          };
        })
      );
      
      return res.json(menu);
    } catch (error) {
      console.error("Error fetching complete menu:", error);
      return res.status(500).json({ message: "Failed to fetch complete menu" });
    }
  });
  
  // Bulk price update
  app.post("/api/menu-items/bulk-update", async (req, res) => {
    // Ensure user is authenticated to perform this action
    if (!req.isAuthenticated()) {
      return res.status(401).json({ message: "Unauthorized" });
    }
    
    try {
      const { categoryId, operation, percentage } = req.body;
      
      // Input validation
      if (!operation || !percentage || isNaN(Number(percentage)) || 
          Number(percentage) < 0 || Number(percentage) > 100 ||
          (operation !== "increase" && operation !== "decrease")) {
        return res.status(400).json({ message: "Invalid parameters" });
      }
      
      const percent = Number(percentage);
      
      // Get the menu items to update based on category filter
      let menuItems;
      if (categoryId && categoryId !== "all") {
        menuItems = await storage.getMenuItemsByCategory(Number(categoryId));
      } else {
        menuItems = await storage.getMenuItems();
      }
      
      if (menuItems.length === 0) {
        return res.status(404).json({ message: "No menu items found matching criteria" });
      }
      
      // Update each menu item price
      const updatedItems = await Promise.all(
        menuItems.map(async (item) => {
          // Parse current price, remove any non-numeric characters except decimal point
          const priceMatch = item.price.match(/(\d+(\.\d+)?)/);
          if (!priceMatch) return null;
          
          const currentPrice = parseFloat(priceMatch[0]);
          if (isNaN(currentPrice)) return null;
          
          // Calculate new price
          let newPrice;
          if (operation === "increase") {
            newPrice = currentPrice * (1 + percent / 100);
          } else {
            newPrice = currentPrice * (1 - percent / 100);
          }
          
          // Round to 2 decimal places and format with currency if present
          const roundedPrice = Math.round(newPrice * 100) / 100;
          const formattedPrice = roundedPrice.toFixed(2) + 
                               (item.price.includes("TL") ? " TL" : "");
          
          // Update the item in storage
          return await storage.updateMenuItem(item.id, { price: formattedPrice });
        })
      );
      
      // Filter out null responses (items that couldn't be updated)
      const successfulUpdates = updatedItems.filter(Boolean);
      
      return res.json({ 
        success: true, 
        updated: successfulUpdates.length,
        message: `${successfulUpdates.length} menü öğesinin fiyatı başarıyla güncellendi`
      });
    } catch (error) {
      console.error("Error in bulk price update:", error);
      return res.status(500).json({ message: "Failed to update prices" });
    }
  });

  // SEO API routes
  app.get("/api/seo", async (req, res) => {
    try {
      const seoSettings = await storage.getSeoSettings();
      if (!seoSettings) {
        return res.status(404).json({ message: "SEO settings not found" });
      }
      return res.json(seoSettings);
    } catch (error) {
      console.error("Error fetching SEO settings:", error);
      return res.status(500).json({ message: "Failed to fetch SEO settings" });
    }
  });

  app.get("/api/seo/active", async (req, res) => {
    try {
      const seoSettings = await storage.getSeoSettings();
      if (!seoSettings) {
        return res.status(404).json({ message: "No active SEO settings found" });
      }
      return res.json(seoSettings);
    } catch (error) {
      console.error("Error fetching active SEO settings:", error);
      return res.status(500).json({ message: "Failed to fetch active SEO settings" });
    }
  });

  app.post("/api/seo", async (req, res) => {
    try {
      if (!req.isAuthenticated()) {
        return res.status(401).json({ message: "Unauthorized" });
      }

      const seoSettings = await storage.createSeoSettings(req.body);
      return res.status(201).json(seoSettings);
    } catch (error) {
      console.error("Error creating SEO settings:", error);
      return res.status(500).json({ message: "Failed to create SEO settings" });
    }
  });

  app.patch("/api/seo/:id", async (req, res) => {
    try {
      if (!req.isAuthenticated()) {
        return res.status(401).json({ message: "Unauthorized" });
      }

      const id = parseInt(req.params.id);
      if (isNaN(id)) {
        return res.status(400).json({ message: "Invalid SEO settings ID" });
      }

      const updatedSeoSettings = await storage.updateSeoSettings(id, req.body);
      return res.json(updatedSeoSettings);
    } catch (error: any) {
      if (error.message && error.message.includes("not found")) {
        return res.status(404).json({ message: error.message });
      }
      console.error("Error updating SEO settings:", error);
      return res.status(500).json({ message: "Failed to update SEO settings" });
    }
  });

  app.delete("/api/seo/:id", async (req, res) => {
    try {
      if (!req.isAuthenticated()) {
        return res.status(401).json({ message: "Unauthorized" });
      }

      const id = parseInt(req.params.id);
      if (isNaN(id)) {
        return res.status(400).json({ message: "Invalid SEO settings ID" });
      }

      const success = await storage.deleteSeoSettings(id);
      if (!success) {
        return res.status(404).json({ message: "SEO settings not found" });
      }
      return res.status(204).end();
    } catch (error) {
      console.error("Error deleting SEO settings:", error);
      return res.status(500).json({ message: "Failed to delete SEO settings" });
    }
  });

  // Payment system routes
  const paymentService = new PaymentService();

  // Create order and initiate payment
  app.post("/api/orders", async (req, res) => {
    try {
      const orderData = insertOrderSchema.parse(req.body);
      const { items, paymentMethod } = req.body;
      
      if (!items || !Array.isArray(items) || items.length === 0) {
        return res.status(400).json({ message: "Sipariş öğeleri gerekli" });
      }

      if (!paymentMethod || !['paytr', 'iyzico', 'cash'].includes(paymentMethod)) {
        return res.status(400).json({ message: "Geçerli ödeme yöntemi seçin" });
      }

      // Generate unique order number
      const orderNumber = `JKS${Date.now()}${Math.floor(Math.random() * 1000)}`;
      
      // Create order
      const order = await storage.createOrder({
        ...orderData,
        orderNumber
      });

      // Create order items
      for (const item of items) {
        await storage.createOrderItem({
          orderId: order.id,
          menuItemId: item.menuItemId,
          quantity: item.quantity,
          price: item.price,
          notes: item.notes
        });
      }

      // Handle payment
      if (paymentMethod === 'cash') {
        // Nakit ödeme - sipariş onaylandı olarak işaretle
        return res.json({
          success: true,
          order,
          paymentMethod: 'cash',
          message: 'Nakit ödeme seçildi. Sipariş alındı.'
        });
      }

      // Online payment
      const paymentRequest = {
        orderId: order.id,
        amount: orderData.totalAmount,
        customerName: orderData.customerName || 'Müşteri',
        customerPhone: orderData.customerPhone,
        items: items.map((item: any) => ({
          name: item.name,
          price: item.price,
          quantity: item.quantity
        }))
      };

      const paymentResult = await paymentService.processPayment(paymentMethod as 'paytr' | 'iyzico', paymentRequest);
      
      if (paymentResult.success) {
        // Create payment record
        await storage.createPayment({
          orderId: order.id,
          paymentMethod,
          paymentProvider: paymentMethod,
          amount: orderData.totalAmount,
          currency: 'TRY'
        });

        return res.json({
          success: true,
          order,
          paymentUrl: paymentResult.paymentUrl,
          paymentToken: paymentResult.token
        });
      } else {
        return res.status(400).json({
          success: false,
          message: paymentResult.errorMessage || 'Ödeme başlatılamadı'
        });
      }

    } catch (error) {
      console.error("Error creating order:", error);
      return res.status(500).json({ message: "Sipariş oluşturulamadı" });
    }
  });

  // Get orders
  app.get("/api/orders", async (req, res) => {
    try {
      if (!req.isAuthenticated()) {
        return res.status(401).json({ message: "Unauthorized" });
      }

      const orders = await storage.getOrders();
      return res.json(orders);
    } catch (error) {
      console.error("Error fetching orders:", error);
      return res.status(500).json({ message: "Failed to fetch orders" });
    }
  });

  // Get single order
  app.get("/api/orders/:id", async (req, res) => {
    try {
      const id = parseInt(req.params.id);
      if (isNaN(id)) {
        return res.status(400).json({ message: "Invalid order ID" });
      }

      const order = await storage.getOrder(id);
      if (!order) {
        return res.status(404).json({ message: "Order not found" });
      }

      return res.json(order);
    } catch (error) {
      console.error("Error fetching order:", error);
      return res.status(500).json({ message: "Failed to fetch order" });
    }
  });

  // Update order status
  app.patch("/api/orders/:id/status", async (req, res) => {
    try {
      if (!req.isAuthenticated()) {
        return res.status(401).json({ message: "Unauthorized" });
      }

      const id = parseInt(req.params.id);
      const { status } = req.body;

      if (isNaN(id)) {
        return res.status(400).json({ message: "Invalid order ID" });
      }

      if (!status || !['pending', 'confirmed', 'preparing', 'ready', 'delivered', 'cancelled'].includes(status)) {
        return res.status(400).json({ message: "Invalid order status" });
      }

      const updatedOrder = await storage.updateOrderStatus(id, status);
      return res.json(updatedOrder);
    } catch (error) {
      console.error("Error updating order status:", error);
      return res.status(500).json({ message: "Failed to update order status" });
    }
  });

  // PayTR payment callback
  app.post("/api/payment/paytr/callback", async (req, res) => {
    try {
      const isValid = paymentService.verifyPayment('paytr', req.body);
      
      if (isValid && req.body.status === 'success') {
        // Payment successful - update order and payment status
        const orderId = parseInt(req.body.merchant_oid.split('_')[1]);
        
        await storage.updateOrderStatus(orderId, 'confirmed');
        // Update payment status logic here
        
        return res.send('OK');
      } else {
        console.error('PayTR callback verification failed or payment failed');
        return res.status(400).send('FAIL');
      }
    } catch (error) {
      console.error("PayTR callback error:", error);
      return res.status(500).send('ERROR');
    }
  });

  // Iyzico payment callback
  app.post("/api/payment/iyzico/callback", async (req, res) => {
    try {
      const { token } = req.body;
      const isValid = paymentService.verifyPayment('iyzico', { token });
      
      if (isValid) {
        // Payment successful - update order status
        // Extract order ID from token or conversation ID
        // Update order and payment status
        
        return res.json({ success: true });
      } else {
        return res.status(400).json({ success: false });
      }
    } catch (error) {
      console.error("Iyzico callback error:", error);
      return res.status(500).json({ success: false });
    }
  });

  // Mock payment page for testing
  app.get("/mock-payment", (req, res) => {
    const { provider, order } = req.query;
    
    res.send(`
      <!DOCTYPE html>
      <html>
      <head>
        <title>Test Ödeme - ${provider}</title>
        <meta charset="utf-8">
        <style>
          body { font-family: Arial, sans-serif; padding: 20px; text-align: center; }
          .container { max-width: 400px; margin: 0 auto; }
          .button { padding: 15px 30px; margin: 10px; font-size: 16px; cursor: pointer; border: none; border-radius: 5px; }
          .success { background: #28a745; color: white; }
          .fail { background: #dc3545; color: white; }
        </style>
      </head>
      <body>
        <div class="container">
          <h2>Test Ödeme Sayfası</h2>
          <p><strong>Ödeme Sağlayıcısı:</strong> ${provider}</p>
          <p><strong>Sipariş No:</strong> ${order}</p>
          <p>Bu test sayfasıdır. Gerçek ödeme işlemi yapılmayacaktır.</p>
          
          <button class="button success" onclick="simulateSuccess()">
            Ödeme Başarılı
          </button>
          <button class="button fail" onclick="simulateFail()">
            Ödeme Başarısız
          </button>
        </div>
        
        <script>
          function simulateSuccess() {
            alert('Ödeme başarılı! Ana sayfaya yönlendiriliyorsunuz...');
            window.location.href = '/';
          }
          
          function simulateFail() {
            alert('Ödeme başarısız! Ana sayfaya yönlendiriliyorsunuz...');
            window.location.href = '/';
          }
        </script>
      </body>
      </html>
    `);
  });

  // Customer Management API Routes
  app.get("/api/customers", async (req, res) => {
    try {
      if (!req.isAuthenticated()) {
        return res.status(401).json({ message: "Unauthorized" });
      }
      const customers = await storage.getCustomers();
      return res.json(customers);
    } catch (error) {
      console.error("Error fetching customers:", error);
      return res.status(500).json({ message: "Failed to fetch customers" });
    }
  });

  app.post("/api/customers", async (req, res) => {
    try {
      if (!req.isAuthenticated()) {
        return res.status(401).json({ message: "Unauthorized" });
      }
      const customer = await storage.createCustomer(req.body);
      return res.status(201).json(customer);
    } catch (error) {
      console.error("Error creating customer:", error);
      return res.status(500).json({ message: "Failed to create customer" });
    }
  });

  app.patch("/api/customers/:id", async (req, res) => {
    try {
      if (!req.isAuthenticated()) {
        return res.status(401).json({ message: "Unauthorized" });
      }
      const id = parseInt(req.params.id);
      const customer = await storage.updateCustomer(id, req.body);
      return res.json(customer);
    } catch (error) {
      console.error("Error updating customer:", error);
      return res.status(500).json({ message: "Failed to update customer" });
    }
  });

  app.delete("/api/customers/:id", async (req, res) => {
    try {
      if (!req.isAuthenticated()) {
        return res.status(401).json({ message: "Unauthorized" });
      }
      const id = parseInt(req.params.id);
      await storage.deleteCustomer(id);
      return res.json({ success: true });
    } catch (error) {
      console.error("Error deleting customer:", error);
      return res.status(500).json({ message: "Failed to delete customer" });
    }
  });

  // Coupon Management API Routes
  app.get("/api/coupons", async (req, res) => {
    try {
      if (!req.isAuthenticated()) {
        return res.status(401).json({ message: "Unauthorized" });
      }
      const coupons = await storage.getCoupons();
      return res.json(coupons);
    } catch (error) {
      console.error("Error fetching coupons:", error);
      return res.status(500).json({ message: "Failed to fetch coupons" });
    }
  });

  app.post("/api/coupons", async (req, res) => {
    try {
      if (!req.isAuthenticated()) {
        return res.status(401).json({ message: "Unauthorized" });
      }
      const coupon = await storage.createCoupon(req.body);
      return res.status(201).json(coupon);
    } catch (error) {
      console.error("Error creating coupon:", error);
      return res.status(500).json({ message: "Failed to create coupon" });
    }
  });

  app.patch("/api/coupons/:id", async (req, res) => {
    try {
      if (!req.isAuthenticated()) {
        return res.status(401).json({ message: "Unauthorized" });
      }
      const id = parseInt(req.params.id);
      const coupon = await storage.updateCoupon(id, req.body);
      return res.json(coupon);
    } catch (error) {
      console.error("Error updating coupon:", error);
      return res.status(500).json({ message: "Failed to update coupon" });
    }
  });

  app.delete("/api/coupons/:id", async (req, res) => {
    try {
      if (!req.isAuthenticated()) {
        return res.status(401).json({ message: "Unauthorized" });
      }
      const id = parseInt(req.params.id);
      await storage.deleteCoupon(id);
      return res.json({ success: true });
    } catch (error) {
      console.error("Error deleting coupon:", error);
      return res.status(500).json({ message: "Failed to delete coupon" });
    }
  });

  // Review Management API Routes
  app.get("/api/reviews", async (req, res) => {
    try {
      if (!req.isAuthenticated()) {
        return res.status(401).json({ message: "Unauthorized" });
      }
      const reviews = await storage.getReviews();
      return res.json(reviews);
    } catch (error) {
      console.error("Error fetching reviews:", error);
      return res.status(500).json({ message: "Failed to fetch reviews" });
    }
  });

  app.patch("/api/reviews/:id", async (req, res) => {
    try {
      if (!req.isAuthenticated()) {
        return res.status(401).json({ message: "Unauthorized" });
      }
      const id = parseInt(req.params.id);
      const review = await storage.updateReview(id, req.body);
      return res.json(review);
    } catch (error) {
      console.error("Error updating review:", error);
      return res.status(500).json({ message: "Failed to update review" });
    }
  });

  app.delete("/api/reviews/:id", async (req, res) => {
    try {
      if (!req.isAuthenticated()) {
        return res.status(401).json({ message: "Unauthorized" });
      }
      const id = parseInt(req.params.id);
      await storage.deleteReview(id);
      return res.json({ success: true });
    } catch (error) {
      console.error("Error deleting review:", error);
      return res.status(500).json({ message: "Failed to delete review" });
    }
  });

  // Analytics API Route
  app.get("/api/analytics", async (req, res) => {
    try {
      if (!req.isAuthenticated()) {
        return res.status(401).json({ message: "Unauthorized" });
      }
      
      // Mock analytics data for now - replace with real data when available
      const analytics = {
        totalOrders: 156,
        totalRevenue: "12,450.75",
        totalCustomers: 89,
        averageOrderValue: "79.85",
        popularItems: [
          { name: "Türk Kahvesi", orders: 45, revenue: "675.00" },
          { name: "Menemen", orders: 38, revenue: "950.00" },
          { name: "Künefe", orders: 32, revenue: "800.00" },
        ],
        recentOrders: [],
        monthlyStats: [
          { month: "Oca", orders: 120, revenue: 8500 },
          { month: "Şub", orders: 145, revenue: 9200 },
          { month: "Mar", orders: 156, revenue: 12450 },
        ],
        customerStats: { newCustomers: 23, returningCustomers: 66 }
      };
      
      return res.json(analytics);
    } catch (error) {
      console.error("Error fetching analytics:", error);
      return res.status(500).json({ message: "Failed to fetch analytics" });
    }
  });

  const httpServer = createServer(app);

  return httpServer;
}
