import { scrypt, randomBytes } from "crypto";
import { promisify } from "util";
import { db } from "./db";
import {
  users,
  categories,
  menuItems,
  restaurants,
  themeSettings,
  seoSettings,
  images,
  menuItemImages,
  orders,
  orderItems,
  customers,
  coupons,
  analytics,
  notifications,
  reviews,
  payments,
} from "@shared/schema";

const scryptAsync = promisify(scrypt);

async function hashPassword(password: string) {
  const salt = randomBytes(16).toString("hex");
  const buf = (await scryptAsync(password, salt, 64)) as Buffer;
  return `${buf.toString("hex")}.${salt}`;
}

export async function runMigrations() {
  try {
    console.log("Starting database setup and migration...");
    
    // Insert default data
    await insertDefaultData();
    
    console.log("✓ Database migration completed successfully");
  } catch (error) {
    console.error("✗ Migration failed:", error);
    throw error;
  }
}

async function insertDefaultData() {
  try {
    // Check if admin user exists
    const existingUser = await db.select().from(users).limit(1);
    if (existingUser.length === 0) {
      // Create admin user
      await db.insert(users).values({
        username: 'admin',
        password: await hashPassword('admin123'),
      });
      console.log("✓ Admin user created");
    }

    // Categories will be added manually through admin panel
    console.log("✓ Categories to be added manually through admin panel");

    // Menu items will be added manually through admin panel
    console.log("✓ Menu items to be added manually through admin panel");

    // Check if restaurant info exists
    const existingRestaurant = await db.select().from(restaurants).limit(1);
    if (existingRestaurant.length === 0) {
      await db.insert(restaurants).values({
        name: 'Jouska Cafe & Restaurant',
        description: 'Geleneksel Türk mutfağından modern yorumlar ve taze kahve seçenekleri',
        address: 'Merkez Mah. Atatürk Cad. No:123 Ankara',
        phone: '+90 312 123 45 67',
        email: 'info@jouska.com',
        website: 'https://jouska.com',
        logo: null,
        isActive: true,
      });
      console.log("✓ Restaurant info created");
    }

    // Check if theme exists
    const existingTheme = await db.select().from(themeSettings).limit(1);
    if (existingTheme.length === 0) {
      await db.insert(themeSettings).values({
        name: 'Varsayılan Tema',
        primaryColor: '#8B4513',
        secondaryColor: '#D2691E',
        backgroundColor: '#F5F5DC',
        textColor: '#2F4F4F',
        fontFamily: 'Inter',
        isActive: true,
      });
      console.log("✓ Theme settings created");
    }

    // Check if SEO settings exist
    const existingSeo = await db.select().from(seoSettings).limit(1);
    if (existingSeo.length === 0) {
      await db.insert(seoSettings).values({
        pageTitle: 'Jouska Cafe & Restaurant - Ankara',
        metaDescription: 'Ankara\'nın en lezzetli kahve ve yemek deneyimi. Geleneksel Türk mutfağı ve modern cafe kültürü bir arada.',
        metaKeywords: 'ankara cafe, restaurant, türk mutfağı, kahve',
        ogTitle: 'Jouska Cafe & Restaurant',
        ogDescription: 'Ankara\'da lezzetin adresi - Jouska Cafe & Restaurant',
        ogImage: null,
        isActive: true,
      });
      console.log("✓ SEO settings created");
    }

    console.log("✓ All default data created successfully");
  } catch (error) {
    console.error("Error inserting default data:", error);
    throw error;
  }
}