import { IStorage } from "./storage";
import { db } from "./db";
import {
  users,
  categories,
  menuItems,
  restaurants,
  themeSettings,
  images,
  menuItemImages,
  seoSettings,
  orders,
  orderItems,
  payments,
  customers,
  coupons,
  analytics,
  notifications,
  reviews,
  type User,
  type InsertUser,
  type Category,
  type InsertCategory,
  type MenuItem,
  type InsertMenuItem,
  type Restaurant,
  type InsertRestaurant,
  type ThemeSettings,
  type InsertThemeSettings,
  type Customer,
  type InsertCustomer,
  type Coupon,
  type InsertCoupon,
  type Review,
  type InsertReview,
  type Analytics,
  type InsertAnalytics,
  type Notification,
  type InsertNotification,
  type Order,
  type InsertOrder,
  type OrderItem,
  type InsertOrderItem,
  type Payment,
  type InsertPayment,
  type Image,
  type InsertImage,
  type MenuItemImage,
  type InsertMenuItemImage,
  type SeoSettings,
  type InsertSeoSettings,
} from "@shared/schema";
import { eq, desc, and, sql } from "drizzle-orm";
import session from "express-session";
import createMemoryStore from "memorystore";

const MemoryStore = createMemoryStore(session);

export class DatabaseStorage implements IStorage {
  public sessionStore: session.Store;

  constructor() {
    this.sessionStore = new MemoryStore({
      checkPeriod: 86400000,
    });
    this.initializeDatabase();
  }

  private async initializeDatabase() {
    // Database initialization is handled by migrate-mysql.ts
    console.log("MySQL database connection established");
  }

  // User methods
  async getUser(id: number): Promise<User | undefined> {
    const [user] = await db.select().from(users).where(eq(users.id, id));
    return user || undefined;
  }

  async getUserByUsername(username: string): Promise<User | undefined> {
    const [user] = await db.select().from(users).where(eq(users.username, username));
    return user || undefined;
  }

  async createUser(insertUser: InsertUser): Promise<User> {
    const result = await db.insert(users).values(insertUser);
    const [user] = await db.select().from(users).where(eq(users.id, result[0].insertId));
    return user;
  }

  // Category methods
  async getCategories(): Promise<Category[]> {
    return await db.select().from(categories).orderBy(categories.order);
  }

  async getCategoryBySlug(slug: string): Promise<Category | undefined> {
    const [category] = await db.select().from(categories).where(eq(categories.slug, slug));
    return category || undefined;
  }

  async createCategory(category: InsertCategory): Promise<Category> {
    const result = await db.insert(categories).values(category);
    const [newCategory] = await db.select().from(categories).where(eq(categories.id, result[0].insertId));
    return newCategory;
  }

  async updateCategory(id: number, category: Partial<InsertCategory>): Promise<Category> {
    await db.update(categories).set(category).where(eq(categories.id, id));
    const [updatedCategory] = await db.select().from(categories).where(eq(categories.id, id));
    return updatedCategory;
  }

  async deleteCategory(id: number): Promise<boolean> {
    const result = await db.delete(categories).where(eq(categories.id, id));
    return result[0].affectedRows > 0;
  }

  // Menu item methods
  async getMenuItems(): Promise<MenuItem[]> {
    return await db.select().from(menuItems).orderBy(menuItems.sortOrder);
  }

  async getMenuItem(id: number): Promise<MenuItem | undefined> {
    const [menuItem] = await db.select().from(menuItems).where(eq(menuItems.id, id));
    return menuItem || undefined;
  }

  async getMenuItemsByCategory(categoryId: number): Promise<MenuItem[]> {
    return await db.select().from(menuItems).where(eq(menuItems.categoryId, categoryId)).orderBy(menuItems.sortOrder);
  }

  async createMenuItem(menuItem: InsertMenuItem): Promise<MenuItem> {
    const result = await db.insert(menuItems).values(menuItem);
    const [newMenuItem] = await db.select().from(menuItems).where(eq(menuItems.id, result[0].insertId));
    return newMenuItem;
  }

  async updateMenuItem(id: number, menuItem: Partial<InsertMenuItem>): Promise<MenuItem> {
    await db.update(menuItems).set(menuItem).where(eq(menuItems.id, id));
    const [updatedMenuItem] = await db.select().from(menuItems).where(eq(menuItems.id, id));
    return updatedMenuItem;
  }

  async deleteMenuItem(id: number): Promise<boolean> {
    const result = await db.delete(menuItems).where(eq(menuItems.id, id));
    return result[0].affectedRows > 0;
  }

  // Restaurant methods
  async getRestaurantInfo(): Promise<Restaurant | undefined> {
    const [restaurant] = await db.select().from(restaurants).limit(1);
    return restaurant || undefined;
  }

  async createRestaurantInfo(info: InsertRestaurant): Promise<Restaurant> {
    const result = await db.insert(restaurants).values(info);
    const [restaurant] = await db.select().from(restaurants).where(eq(restaurants.id, result[0].insertId));
    return restaurant;
  }

  async updateRestaurantInfo(id: number, info: Partial<InsertRestaurant>): Promise<Restaurant> {
    await db.update(restaurants).set(info).where(eq(restaurants.id, id));
    const [updatedRestaurant] = await db.select().from(restaurants).where(eq(restaurants.id, id));
    return updatedRestaurant;
  }

  // Theme methods
  async getThemeSettings(): Promise<ThemeSettings | undefined> {
    const [theme] = await db.select().from(themeSettings).limit(1);
    return theme || undefined;
  }

  async getActiveTheme(): Promise<ThemeSettings | undefined> {
    const [theme] = await db.select().from(themeSettings).where(eq(themeSettings.isActive, true));
    return theme || undefined;
  }

  async createThemeSettings(settings: InsertThemeSettings): Promise<ThemeSettings> {
    const result = await db.insert(themeSettings).values(settings);
    const [theme] = await db.select().from(themeSettings).where(eq(themeSettings.id, result[0].insertId));
    return theme;
  }

  async updateThemeSettings(id: number, settings: Partial<InsertThemeSettings>): Promise<ThemeSettings> {
    await db.update(themeSettings).set(settings).where(eq(themeSettings.id, id));
    const [updatedTheme] = await db.select().from(themeSettings).where(eq(themeSettings.id, id));
    return updatedTheme;
  }

  async activateTheme(id: number): Promise<ThemeSettings> {
    // Deactivate all themes first
    await db.update(themeSettings).set({ isActive: false });
    // Activate the selected theme
    await db.update(themeSettings).set({ isActive: true }).where(eq(themeSettings.id, id));
    const [theme] = await db.select().from(themeSettings).where(eq(themeSettings.id, id));
    return theme;
  }

  async deleteThemeSettings(id: number): Promise<boolean> {
    const result = await db.delete(themeSettings).where(eq(themeSettings.id, id));
    return result[0].affectedRows > 0;
  }

  // Image methods
  async getImages(): Promise<Image[]> {
    return await db.select().from(images).orderBy(desc(images.createdAt));
  }

  async getImage(id: number): Promise<Image | undefined> {
    const [image] = await db.select().from(images).where(eq(images.id, id));
    return image || undefined;
  }

  async createImage(image: InsertImage): Promise<Image> {
    const result = await db.insert(images).values(image);
    const [newImage] = await db.select().from(images).where(eq(images.id, result[0].insertId));
    return newImage;
  }

  async updateImage(id: number, image: Partial<InsertImage>): Promise<Image> {
    await db.update(images).set(image).where(eq(images.id, id));
    const [updatedImage] = await db.select().from(images).where(eq(images.id, id));
    return updatedImage;
  }

  async deleteImage(id: number): Promise<boolean> {
    const result = await db.delete(images).where(eq(images.id, id));
    return result[0].affectedRows > 0;
  }

  async getMenuItemImages(menuItemId: number): Promise<Image[]> {
    const result = await db
      .select({
        id: images.id,
        url: images.url,
        alt: images.alt,
        menuItemId: images.menuItemId,
        createdAt: images.createdAt,
      })
      .from(menuItemImages)
      .innerJoin(images, eq(menuItemImages.imageId, images.id))
      .where(eq(menuItemImages.menuItemId, menuItemId))
      .orderBy(menuItemImages.order);

    return result;
  }

  async addImageToMenuItem(menuItemId: number, imageId: number, order: number = 0): Promise<MenuItemImage> {
    const result = await db.insert(menuItemImages).values({
      menuItemId,
      imageId,
      order,
    });
    const [menuItemImage] = await db.select().from(menuItemImages).where(eq(menuItemImages.id, result[0].insertId));
    return menuItemImage;
  }

  async removeImageFromMenuItem(menuItemId: number, imageId: number): Promise<boolean> {
    const result = await db.delete(menuItemImages)
      .where(and(eq(menuItemImages.menuItemId, menuItemId), eq(menuItemImages.imageId, imageId)));
    return result[0].affectedRows > 0;
  }

  // SEO methods
  async getSeoSettings(): Promise<SeoSettings | undefined> {
    const [seo] = await db.select().from(seoSettings).where(eq(seoSettings.isActive, true));
    return seo || undefined;
  }

  async createSeoSettings(settings: InsertSeoSettings): Promise<SeoSettings> {
    const result = await db.insert(seoSettings).values(settings);
    const [seo] = await db.select().from(seoSettings).where(eq(seoSettings.id, result[0].insertId));
    return seo;
  }

  async updateSeoSettings(id: number, settings: Partial<InsertSeoSettings>): Promise<SeoSettings> {
    await db.update(seoSettings).set(settings).where(eq(seoSettings.id, id));
    const [updatedSeo] = await db.select().from(seoSettings).where(eq(seoSettings.id, id));
    return updatedSeo;
  }

  async deleteSeoSettings(id: number): Promise<boolean> {
    const result = await db.delete(seoSettings).where(eq(seoSettings.id, id));
    return result[0].affectedRows > 0;
  }

  // Order methods
  async getOrders(): Promise<Order[]> {
    return await db.select().from(orders).orderBy(desc(orders.createdAt));
  }

  async getOrder(id: number): Promise<Order | undefined> {
    const [order] = await db.select().from(orders).where(eq(orders.id, id));
    return order || undefined;
  }

  async getOrdersByStatus(status: string): Promise<Order[]> {
    return await db.select().from(orders).where(eq(orders.status, status)).orderBy(desc(orders.createdAt));
  }

  async createOrder(order: InsertOrder): Promise<Order> {
    const orderNumber = `ORD-${Date.now()}`;
    const orderData = { ...order, orderNumber };
    const result = await db.insert(orders).values(orderData);
    
    // Create notification for new order
    await db.insert(notifications).values({
      title: "Yeni Sipariş",
      message: `Yeni sipariş alındı: ${orderNumber}`,
      type: "order",
      targetAudience: "admin",
    });

    const [newOrder] = await db.select().from(orders).where(eq(orders.id, result[0].insertId));
    return newOrder;
  }

  async updateOrder(id: number, order: Partial<InsertOrder>): Promise<Order> {
    await db.update(orders).set(order).where(eq(orders.id, id));
    const [updatedOrder] = await db.select().from(orders).where(eq(orders.id, id));
    return updatedOrder;
  }

  async updateOrderStatus(id: number, status: string): Promise<Order> {
    await db.update(orders).set({ status }).where(eq(orders.id, id));
    const [updatedOrder] = await db.select().from(orders).where(eq(orders.id, id));
    return updatedOrder;
  }

  async deleteOrder(id: number): Promise<boolean> {
    const result = await db.delete(orders).where(eq(orders.id, id));
    return result[0].affectedRows > 0;
  }

  // Order item methods
  async getOrderItems(orderId: number): Promise<OrderItem[]> {
    return await db.select().from(orderItems).where(eq(orderItems.orderId, orderId));
  }

  async createOrderItem(orderItem: InsertOrderItem): Promise<OrderItem> {
    const result = await db.insert(orderItems).values(orderItem);
    const [newOrderItem] = await db.select().from(orderItems).where(eq(orderItems.id, result[0].insertId));
    return newOrderItem;
  }

  async updateOrderItem(id: number, orderItem: Partial<InsertOrderItem>): Promise<OrderItem> {
    await db.update(orderItems).set(orderItem).where(eq(orderItems.id, id));
    const [updatedOrderItem] = await db.select().from(orderItems).where(eq(orderItems.id, id));
    return updatedOrderItem;
  }

  async deleteOrderItem(id: number): Promise<boolean> {
    const result = await db.delete(orderItems).where(eq(orderItems.id, id));
    return result[0].affectedRows > 0;
  }

  // Payment methods
  async getPayments(): Promise<Payment[]> {
    return await db.select().from(payments).orderBy(desc(payments.createdAt));
  }

  async getPayment(id: number): Promise<Payment | undefined> {
    const [payment] = await db.select().from(payments).where(eq(payments.id, id));
    return payment || undefined;
  }

  async getPaymentsByOrder(orderId: number): Promise<Payment[]> {
    return await db.select().from(payments).where(eq(payments.orderId, orderId));
  }

  async createPayment(payment: InsertPayment): Promise<Payment> {
    const result = await db.insert(payments).values(payment);
    const [newPayment] = await db.select().from(payments).where(eq(payments.id, result[0].insertId));
    return newPayment;
  }

  async updatePayment(id: number, payment: Partial<InsertPayment>): Promise<Payment> {
    await db.update(payments).set(payment).where(eq(payments.id, id));
    const [updatedPayment] = await db.select().from(payments).where(eq(payments.id, id));
    return updatedPayment;
  }

  async deletePayment(id: number): Promise<boolean> {
    const result = await db.delete(payments).where(eq(payments.id, id));
    return result[0].affectedRows > 0;
  }

  // Customer methods
  async getCustomers(): Promise<Customer[]> {
    return await db.select().from(customers).orderBy(desc(customers.createdAt));
  }

  async getCustomer(id: number): Promise<Customer | undefined> {
    const [customer] = await db.select().from(customers).where(eq(customers.id, id));
    return customer || undefined;
  }

  async getCustomerByPhone(phone: string): Promise<Customer | undefined> {
    const [customer] = await db.select().from(customers).where(eq(customers.phone, phone));
    return customer || undefined;
  }

  async createCustomer(customer: InsertCustomer): Promise<Customer> {
    const result = await db.insert(customers).values(customer);
    const [newCustomer] = await db.select().from(customers).where(eq(customers.id, result[0].insertId));
    return newCustomer;
  }

  async updateCustomer(id: number, customer: Partial<InsertCustomer>): Promise<Customer> {
    await db.update(customers).set(customer).where(eq(customers.id, id));
    const [updatedCustomer] = await db.select().from(customers).where(eq(customers.id, id));
    return updatedCustomer;
  }

  async deleteCustomer(id: number): Promise<boolean> {
    const result = await db.delete(customers).where(eq(customers.id, id));
    return result[0].affectedRows > 0;
  }

  async addCustomerPoints(id: number, points: number): Promise<Customer> {
    await db.update(customers).set({ points: sql`${customers.points} + ${points}` }).where(eq(customers.id, id));
    const [customer] = await db.select().from(customers).where(eq(customers.id, id));
    return customer;
  }

  // Coupon methods
  async getCoupons(): Promise<Coupon[]> {
    return await db.select().from(coupons).orderBy(desc(coupons.createdAt));
  }

  async getCoupon(id: number): Promise<Coupon | undefined> {
    const [coupon] = await db.select().from(coupons).where(eq(coupons.id, id));
    return coupon || undefined;
  }

  async getCouponByCode(code: string): Promise<Coupon | undefined> {
    const [coupon] = await db.select().from(coupons).where(eq(coupons.code, code));
    return coupon || undefined;
  }

  async createCoupon(coupon: InsertCoupon): Promise<Coupon> {
    const result = await db.insert(coupons).values(coupon);
    const [newCoupon] = await db.select().from(coupons).where(eq(coupons.id, result[0].insertId));
    return newCoupon;
  }

  async updateCoupon(id: number, coupon: Partial<InsertCoupon>): Promise<Coupon> {
    await db.update(coupons).set(coupon).where(eq(coupons.id, id));
    const [updatedCoupon] = await db.select().from(coupons).where(eq(coupons.id, id));
    return updatedCoupon;
  }

  async useCoupon(id: number): Promise<Coupon> {
    await db.update(coupons).set({ usedCount: sql`${coupons.usedCount} + 1` }).where(eq(coupons.id, id));
    const [coupon] = await db.select().from(coupons).where(eq(coupons.id, id));
    return coupon;
  }

  async deleteCoupon(id: number): Promise<boolean> {
    const result = await db.delete(coupons).where(eq(coupons.id, id));
    return result[0].affectedRows > 0;
  }

  // Analytics methods
  async getAnalytics(limit: number = 100): Promise<Analytics[]> {
    return await db.select().from(analytics).orderBy(desc(analytics.timestamp)).limit(limit);
  }

  async createAnalyticsEvent(analyticsData: InsertAnalytics): Promise<Analytics> {
    const result = await db.insert(analytics).values(analyticsData);
    const [analytics_record] = await db.select().from(analytics).where(eq(analytics.id, result[0].insertId));
    return analytics_record;
  }

  async getAnalyticsByEvent(event: string, limit: number = 100): Promise<Analytics[]> {
    return await db.select().from(analytics).where(eq(analytics.event, event)).orderBy(desc(analytics.timestamp)).limit(limit);
  }

  // Notification methods
  async getNotifications(): Promise<Notification[]> {
    return await db.select().from(notifications).orderBy(desc(notifications.createdAt));
  }

  async getUnreadNotifications(): Promise<Notification[]> {
    return await db.select().from(notifications).where(eq(notifications.isRead, false)).orderBy(desc(notifications.createdAt));
  }

  async createNotification(notification: InsertNotification): Promise<Notification> {
    const result = await db.insert(notifications).values(notification);
    const [newNotification] = await db.select().from(notifications).where(eq(notifications.id, result[0].insertId));
    return newNotification;
  }

  async markNotificationAsRead(id: number): Promise<Notification> {
    await db.update(notifications).set({ isRead: true }).where(eq(notifications.id, id));
    const [notification] = await db.select().from(notifications).where(eq(notifications.id, id));
    return notification;
  }

  async deleteNotification(id: number): Promise<boolean> {
    const result = await db.delete(notifications).where(eq(notifications.id, id));
    return result[0].affectedRows > 0;
  }

  // Review methods
  async getReviews(): Promise<Review[]> {
    return await db.select().from(reviews).orderBy(desc(reviews.createdAt));
  }

  async getApprovedReviews(): Promise<Review[]> {
    return await db.select().from(reviews).where(eq(reviews.isApproved, true)).orderBy(desc(reviews.createdAt));
  }

  async getMenuItemReviews(menuItemId: number): Promise<Review[]> {
    return await db.select().from(reviews).where(eq(reviews.menuItemId, menuItemId)).orderBy(desc(reviews.createdAt));
  }

  async createReview(review: InsertReview): Promise<Review> {
    const result = await db.insert(reviews).values(review);
    const [newReview] = await db.select().from(reviews).where(eq(reviews.id, result[0].insertId));
    return newReview;
  }

  async updateReview(id: number, review: Partial<InsertReview>): Promise<Review> {
    await db.update(reviews).set(review).where(eq(reviews.id, id));
    const [updatedReview] = await db.select().from(reviews).where(eq(reviews.id, id));
    return updatedReview;
  }

  async approveReview(id: number): Promise<Review> {
    await db.update(reviews).set({ isApproved: true }).where(eq(reviews.id, id));
    const [review] = await db.select().from(reviews).where(eq(reviews.id, id));
    return review;
  }

  async deleteReview(id: number): Promise<boolean> {
    const result = await db.delete(reviews).where(eq(reviews.id, id));
    return result[0].affectedRows > 0;
  }
}

export const storage: IStorage = new DatabaseStorage();