import React, { useState, useEffect } from "react";
import { Header } from "@/components/layout/header";
import { Footer } from "@/components/layout/footer";
import { CategoryNavigation } from "@/components/menu/category-navigation";
import { MenuCategory } from "@/components/menu/menu-category";
import { MenuItemCard } from "@/components/menu/menu-item";
import { QRModal } from "@/components/menu/qr-modal";
import { OfflineBanner } from "@/components/offline-banner";

import { Button } from "@/components/ui/button";
import { ArrowUp, Utensils, Smartphone, Tablet, ArrowRight, X, Heart, MessageCircle, Star } from "lucide-react";
import { useCategories } from "@/hooks/use-categories";
import { useMenuItems } from "@/hooks/use-menu-items";
import { useQuery, useQueryClient } from "@tanstack/react-query";
import { Restaurant } from "@shared/schema";
import { SectionTitle } from "@/components/ui/section-title";
import { GoogleReviewsButton } from "@/components/ui/google-reviews-button";

export default function Home() {

  const [qrModalOpen, setQrModalOpen] = useState(false);
  const [showBackToTop, setShowBackToTop] = useState(false);

  const [searchQuery, setSearchQuery] = useState('');
  const [showFavorites, setShowFavorites] = useState(false);
  const [isOnline, setIsOnline] = useState(navigator.onLine);
  const [selectedCategory, setSelectedCategory] = useState<string | null>(null);
  
  const queryClient = useQueryClient();
  const [favorites, setFavorites] = useState<number[]>([]);
  const [searchResults, setSearchResults] = useState<any[]>([]);
  
  // Fetch restaurant information
  const restaurantQuery = useQuery<Restaurant>({ 
    queryKey: ['/api/restaurant']
  });
  
  // Cache restaurant info for offline use
  useEffect(() => {
    if (restaurantQuery.data) {
      import('@/lib/offline-storage').then(({ cacheRestaurantInfo }) => {
        cacheRestaurantInfo(restaurantQuery.data);
      });
    }
  }, [restaurantQuery.data]);
  
  // Fetch menu data
  const { data: categories, isLoading: isCategoriesLoading } = useCategories();
  const { data: menuItems, isLoading: isMenuItemsLoading } = useMenuItems();
  
  const isLoading = isCategoriesLoading || isMenuItemsLoading;
  
  // Handle refresh for online/offline status
  const handleRefresh = () => {
    queryClient.invalidateQueries();
  };


  
  // Monitor online/offline status
  useEffect(() => {
    const handleOnlineStatus = () => {
      setIsOnline(navigator.onLine);
    };
    
    window.addEventListener('online', handleOnlineStatus);
    window.addEventListener('offline', handleOnlineStatus);
    
    return () => {
      window.removeEventListener('online', handleOnlineStatus);
      window.removeEventListener('offline', handleOnlineStatus);
    };
  }, []);
  
  // Handle search functionality with advanced features
  useEffect(() => {
    if ((!menuItems || !categories) || searchQuery.trim() === '') {
      setSearchResults([]);
      return;
    }
    
    const query = searchQuery.toLowerCase().trim();
    
    // Create a map of category IDs to category names for quick lookup
    const categoryMap = new Map();
    categories.forEach(category => {
      categoryMap.set(category.id, {
        name: category.name.toLowerCase(),
        slug: category.slug.toLowerCase()
      });
    });
    
    // Assign scores to each item based on match quality
    const scoredItems = menuItems.map(item => {
      let score = 0;
      
      // Direct name match (highest priority)
      if (item.name.toLowerCase() === query) {
        score += 100;
      } else if (item.name.toLowerCase().includes(query)) {
        score += 50;
      }
      
      // Description match (medium priority)
      if (item.description && item.description.toLowerCase().includes(query)) {
        score += 30;
      }
      
      // Category match (lower priority but still relevant)
      const category = categoryMap.get(item.categoryId);
      if (category) {
        if (category.name === query || category.slug === query) {
          score += 20;
        } else if (category.name.includes(query) || category.slug.includes(query)) {
          score += 10;
        }
      }
      
      // If item is vegetarian and user searches for "vejetaryen" or "veggie"
      if (item.isVegetarian && 
          (query === "vejetaryen" || query === "vegan" || query === "veggie" || query === "sebze")) {
        score += 25;
      }
      
      // If item is spicy and user searches for "acı" or "spicy"
      if (item.isSpicy && (query === "acı" || query === "acili" || query === "baharatlı" || query === "spicy")) {
        score += 25;
      }
      
      // Price match (exact match on price number)
      if (item.price.replace(/\D/g, '').includes(query.replace(/\D/g, ''))) {
        score += 15;
      }
      
      return { item, score };
    });
    
    // Filter items with a score > 0 and sort by score descending
    const filtered = scoredItems
      .filter(({ score }) => score > 0)
      .sort((a, b) => b.score - a.score)
      .map(({ item }) => item);
    
    setSearchResults(filtered);
  }, [searchQuery, menuItems, categories]);
  
  // Load favorites from localStorage
  useEffect(() => {
    if (typeof window !== 'undefined') {
      const storedFavorites = JSON.parse(localStorage.getItem('favorites') || '[]');
      setFavorites(storedFavorites);
    }
  }, []);
  
  // Listen for favorites changes in localStorage
  useEffect(() => {
    const handleStorageChange = () => {
      if (typeof window !== 'undefined') {
        const storedFavorites = JSON.parse(localStorage.getItem('favorites') || '[]');
        setFavorites(storedFavorites);
      }
    };
    
    window.addEventListener('storage', handleStorageChange);
    // Also listen to a custom event for changes made within the same tab
    window.addEventListener('favoritesUpdated', handleStorageChange);
    
    return () => {
      window.removeEventListener('storage', handleStorageChange);
      window.removeEventListener('favoritesUpdated', handleStorageChange);
    };
  }, []);
  
  // Handle scroll for back to top button
  useEffect(() => {
    const handleScroll = () => {
      setShowBackToTop(window.pageYOffset > 300);
    };
    
    window.addEventListener('scroll', handleScroll);
    return () => window.removeEventListener('scroll', handleScroll);
  }, []);
  
  // Handle back to top click
  const scrollToTop = () => {
    window.scrollTo({
      top: 0,
      behavior: 'smooth'
    });
  };
  
  // QR Code URL (chatbot tetikleyicisi ile)
  const menuUrl = typeof window !== 'undefined' 
    ? `${window.location.protocol}//${window.location.host}${window.location.pathname}?qr=true` 
    : '';
  
  return (
    <div className="min-h-screen flex flex-col">
      {/* Offline Banner */}
      <div className="fixed top-0 left-0 right-0 z-[60] pt-1 px-4">
        <OfflineBanner isOnline={isOnline} onRefresh={handleRefresh} />
      </div>
      
      {/* Fixed header that stays on top */}
      <div className="fixed top-0 left-0 right-0 z-50">
        <Header 
          restaurantInfo={restaurantQuery.data}
          onQrButtonClick={() => setQrModalOpen(true)} 
          searchQuery={searchQuery}
          setSearchQuery={setSearchQuery}
        />
        
        {!isLoading && categories && (
          <CategoryNavigation 
            categories={categories} 
            selectedCategory={selectedCategory}
            onCategorySelect={setSelectedCategory}
          />
        )}
      </div>
      
      {/* Spacer to push content below fixed header */}
      <div className="h-[130px]"></div>
      
      <main className="flex-grow container mx-auto px-4 py-6 scroll-smooth">
        
        {/* Menu Welcome Section */}
        {!isLoading && restaurantQuery.data && (
          <div className="mb-12 bg-gradient-to-br from-primary/5 to-accent/5 rounded-2xl p-8 border border-primary/10">
            <div className="flex flex-col md:flex-row items-center justify-between">
              <div className="mb-6 md:mb-0 md:mr-8 flex-1">
                <SectionTitle 
                  title={`${restaurantQuery.data.name} Menü`}
                  subtitle="Özenle hazırlanmış lezzetlerimizi keşfedin"
                  size="xl"
                  decorative={false}
                  titleClassName="heading-gradient mb-3"
                />
                <p className="text-muted-foreground mb-6 max-w-xl">
                  Aşağıdaki kategorilerde sunduğumuz yemeklerimizi inceleyebilir, detaylı bilgi için dilediğiniz yemeğe tıklayabilirsiniz.
                </p>
                
                {/* Mobile search box - only visible on mobile */}
                <div className="relative mb-6 max-w-md md:hidden">
                  <div className="absolute inset-y-0 start-0 flex items-center ps-3 pointer-events-none">
                    <svg className="w-4 h-4 text-primary" aria-hidden="true" xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 20 20">
                      <path stroke="currentColor" strokeLinecap="round" strokeLinejoin="round" strokeWidth="2" d="m19 19-4-4m0-7A7 7 0 1 1 1 8a7 7 0 0 1 14 0Z"/>
                    </svg>
                  </div>
                  <input 
                    id="mobile-search-input"
                    type="search" 
                    className="block w-full p-4 ps-10 text-sm border border-primary/20 rounded-lg bg-background focus:ring-primary focus:border-primary" 
                    placeholder="Menüde ara... (örn: kahvaltı, burger, salata)" 
                    value={searchQuery}
                    onChange={(e) => setSearchQuery(e.target.value)}
                  />
                  {searchQuery && (
                    <button 
                      type="button" 
                      className="absolute end-2.5 bottom-2.5 focus:ring-4 focus:outline-none font-medium rounded-lg text-sm px-3 py-2 text-primary hover:text-primary/80"
                      onClick={() => setSearchQuery('')}
                    >
                      <X className="h-4 w-4" />
                    </button>
                  )}
                </div>
                
                {/* Mobile search keywords - only visible on mobile */}
                <div className="flex flex-wrap gap-2 mb-6 md:hidden">
                  {["kahvaltı", "burger", "tatlı", "vejetaryen"].map((keyword) => (
                    <button
                      key={keyword}
                      onClick={() => setSearchQuery(keyword)}
                      className="px-3 py-1 bg-primary/10 hover:bg-primary/20 rounded-full text-xs text-primary transition-colors"
                    >
                      {keyword}
                    </button>
                  ))}
                </div>
                
                <div className="flex items-center space-x-2 text-sm text-muted-foreground">
                  <Utensils className="h-4 w-4 text-primary" />
                  <span>{categories?.length || 0} kategori</span>
                  <span className="w-1 h-1 rounded-full bg-primary/30"></span>
                  <span>{menuItems?.length || 0} ürün</span>
                </div>
              </div>
              
              <div className="w-24 h-24 md:w-32 md:h-32 relative flex-shrink-0">
                <div className="w-full h-full rounded-full bg-primary/10 flex items-center justify-center">
                  <div className="w-4/5 h-4/5 rounded-full bg-primary/20 flex items-center justify-center">
                    <div className="w-3/5 h-3/5 rounded-full bg-primary/30 flex items-center justify-center">
                      <Utensils className="h-8 w-8 text-primary" />
                    </div>
                  </div>
                </div>
              </div>
            </div>
          </div>
        )}
        


        {/* Favorites */}
        {showFavorites && favorites.length > 0 && !searchQuery.trim() && (
          <div className="mb-10">
            <div className="flex justify-between items-center">
              <SectionTitle 
                title="Favori Menüleriniz" 
                subtitle="Beğendiğiniz ve kaydettiğiniz tüm menü öğeleri"
                size="lg"
              />
              <button
                onClick={() => setShowFavorites(false)}
                className="text-primary hover:text-primary/80 text-sm font-medium flex items-center"
              >
                <X className="h-4 w-4 mr-1" />
                Favorileri Gizle
              </button>
            </div>
            <div className="grid grid-cols-1 sm:grid-cols-2 md:grid-cols-3 lg:grid-cols-4 gap-6">
              {menuItems?.filter(item => favorites.includes(item.id)).map((item) => (
                <MenuItemCard key={item.id} item={item} />
              ))}
            </div>
          </div>
        )}
        
        {/* Favorites Button */}
        {!showFavorites && favorites.length > 0 && !searchQuery.trim() && (
          <div className="mb-6">
            <button
              onClick={() => setShowFavorites(true)}
              className="flex items-center gap-2 px-4 py-2 bg-primary text-white rounded-lg hover:bg-primary/90 transition-colors shadow-sm"
            >
              <Heart className="h-4 w-4 fill-current" />
              <span>Favorileri Göster ({favorites.length})</span>
            </button>
          </div>
        )}
        
        {/* Search Results */}
        {searchQuery.trim() !== '' && searchResults.length > 0 && (
          <div className="mb-10">
            <SectionTitle 
              title="Arama Sonuçları" 
              subtitle={`"${searchQuery}" için ${searchResults.length} sonuç bulundu`}
              size="lg"
            />
            <div className="grid grid-cols-1 sm:grid-cols-2 md:grid-cols-3 lg:grid-cols-4 gap-6">
              {searchResults.map((item) => (
                <div key={`search-${item.id}`} className="animate-fadeIn">
                  {/* We render the menu item directly without using MenuCategory */}
                  <div className="bg-card rounded-xl shadow-sm hover:shadow-md transition-shadow overflow-hidden border border-border/40">
                    {item.imageUrl && (
                      <div className="aspect-video overflow-hidden">
                        <img
                          src={item.imageUrl}
                          alt={item.name}
                          className="w-full h-full object-cover transform hover:scale-105 transition-transform duration-500"
                        />
                      </div>
                    )}
                    <div className="p-4">
                      <h3 className="font-bold text-lg font-display">{item.name}</h3>
                      {item.description && (
                        <p className="text-muted-foreground line-clamp-2 text-sm mt-1">{item.description}</p>
                      )}
                      <div className="mt-3 flex justify-between items-center">
                        <span className="font-bold text-primary">{item.price}</span>
                        <div className="flex items-center gap-1">
                          {item.isVegetarian && (
                            <span className="bg-green-100 text-green-800 text-xs px-2 py-1 rounded-full dark:bg-green-900 dark:text-green-100">
                              Vejetaryen
                            </span>
                          )}
                        </div>
                      </div>
                    </div>
                  </div>
                </div>
              ))}
            </div>
          </div>
        )}
        
        {/* Show "no results" message when search returns nothing */}
        {searchQuery.trim() !== '' && searchResults.length === 0 && (
          <div className="text-center py-12 bg-card/50 rounded-xl mb-10">
            <div className="mx-auto w-16 h-16 mb-4 text-muted-foreground">
              <svg xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={1.5} 
                  d="M21 21l-6-6m2-5a7 7 0 11-14 0 7 7 0 0114 0z" />
              </svg>
            </div>
            <h3 className="text-lg font-bold mb-2">Sonuç Bulunamadı</h3>
            <p className="text-muted-foreground max-w-md mx-auto">
              "{searchQuery}" için uygun sonuç bulunamadı. Farklı bir kelime ile tekrar arayabilirsiniz.
            </p>
            <div className="mt-6">
              <p className="text-sm text-muted-foreground mb-3">Şunları aramayı deneyebilirsiniz:</p>
              <div className="flex flex-wrap justify-center gap-2 max-w-xl mx-auto">
                {["kahvaltı", "burger", "salata", "tatlı", "içecek", "kahve", "vejetaryen"].map((suggestion) => (
                  <button
                    key={suggestion}
                    onClick={() => setSearchQuery(suggestion)}
                    className="px-3 py-1 bg-primary/10 hover:bg-primary/20 rounded-full text-sm text-primary transition-colors"
                  >
                    {suggestion}
                  </button>
                ))}
              </div>
            </div>
          </div>
        )}
        
        {/* Regular menu if no search or search is cleared */}
        {(searchQuery.trim() === '' || !searchResults) && !showFavorites && (
          isLoading ? (
            <div className="flex justify-center items-center py-12">
              <div className="animate-spin rounded-full h-12 w-12 border-b-2 border-primary"></div>
            </div>
          ) : selectedCategory ? (
            // Show only selected category
            categories
              ?.filter(category => category.slug === selectedCategory)
              .map((category) => {
                const categoryItems = menuItems?.filter(item => item.categoryId === category.id && item.isActive) || [];
                if (categoryItems.length === 0) return null;
                
                return (
                  <div key={category.id} className="mb-8">
                    <div className="flex items-center justify-between mb-6">
                      <h2 className="text-2xl font-bold text-primary">{category.name}</h2>
                      <button
                        onClick={() => setSelectedCategory(null)}
                        className="text-primary hover:text-primary/80 text-sm font-medium flex items-center gap-1"
                      >
                        <X className="h-4 w-4" />
                        Tümünü Göster
                      </button>
                    </div>
                    <div className="grid grid-cols-1 sm:grid-cols-2 md:grid-cols-3 lg:grid-cols-4 gap-6">
                      {categoryItems.map((item) => (
                        <MenuItemCard key={item.id} item={item} />
                      ))}
                    </div>
                  </div>
                );
              })
          ) : (
            // Show all categories
            categories?.map(category => {
              const categoryItems = menuItems?.filter(item => item.categoryId === category.id && item.isActive) || [];
              if (categoryItems.length === 0) return null;
              
              return (
                <MenuCategory 
                  key={category.id} 
                  id={category.slug} 
                  title={category.name} 
                  items={categoryItems} 
                />
              );
            })
          )
        )}
      </main>
      
      <Footer restaurantInfo={restaurantQuery.data} />
      
      {/* Back to Top Button */}
      <Button
        onClick={scrollToTop}
        className={`fixed bottom-6 right-6 bg-primary text-white p-3 rounded-full shadow-lg transition-opacity duration-300 hover:bg-opacity-90 ${
          showBackToTop ? 'opacity-100' : 'opacity-0 pointer-events-none'
        }`}
      >
        <ArrowUp className="h-5 w-5" />
      </Button>
      
      {/* Google Reviews Button */}
      <Button
        onClick={() => {
          const googleUrl = restaurantQuery.data?.googleMyBusinessUrl || 
                           (restaurantQuery.data?.googlePlaceId ? 
                             `https://search.google.com/local/writereview?placeid=${restaurantQuery.data.googlePlaceId}` :
                             `https://www.google.com/maps/search/?api=1&query=${encodeURIComponent(restaurantQuery.data?.name || 'Jouska Cafe Restaurant')}`);
          window.open(googleUrl, '_blank');
        }}
        className="fixed bottom-20 left-6 bg-yellow-500 hover:bg-yellow-600 text-white p-3 rounded-full shadow-lg transition-all duration-300 z-30"
        title="Google'da Yorumla"
      >
        <Star className="h-4 w-4 fill-white" />
      </Button>


      

      
      {/* QR Code Modal */}
      <QRModal 
        open={qrModalOpen} 
        onOpenChange={setQrModalOpen} 
        url={menuUrl} 
      />


    </div>
  );
}
