import React from "react";
import { useForm } from "react-hook-form";
import { zodResolver } from "@hookform/resolvers/zod";
import { z } from "zod";
import { useAuth } from "@/hooks/use-auth";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { Redirect } from "wouter";
import { useToast } from "@/hooks/use-toast";

// Validation schema for login form
const loginSchema = z.object({
  username: z.string().min(1, { message: "Kullanıcı adı gerekli" }),
  password: z.string().min(1, { message: "Şifre gerekli" }),
});

type LoginFormData = z.infer<typeof loginSchema>;

const AuthPage = () => {
  const { user, loginMutation } = useAuth();
  const { toast } = useToast();
  
  // Login form
  const loginForm = useForm<LoginFormData>({
    resolver: zodResolver(loginSchema),
    defaultValues: {
      username: "",
      password: "",
    },
  });
  
  // Handle login submission
  const handleLogin = async (data: LoginFormData) => {
    try {
      await loginMutation.mutateAsync(data);
      toast({
        title: "Giriş Başarılı",
        description: "Jouska Yönetim Paneline yönlendiriliyorsunuz.",
      });
    } catch (error) {
      console.error("Login error:", error);
      toast({
        title: "Giriş Başarısız",
        description: "Kullanıcı adı veya şifre hatalı.",
        variant: "destructive"
      });
    }
  };
  
  // Redirect if user is already logged in
  if (user) {
    return <Redirect to="/yonetim-paneli" />;
  }
  
  return (
    <div className="min-h-screen flex flex-col md:flex-row">
      {/* Auth form column */}
      <div className="md:w-1/2 flex items-center justify-center p-8">
        <Card className="w-full max-w-md shadow-lg border-0 rounded-xl overflow-hidden">
          <CardHeader className="bg-primary text-primary-foreground pb-8 pt-10">
            <CardTitle className="text-2xl font-bold text-center">
              Jouska Yönetim Paneli
            </CardTitle>
            <CardDescription className="text-primary-foreground/80 text-center">
              Menü ve içerik yönetimi için giriş yapın
            </CardDescription>
          </CardHeader>
          
          <CardContent className="p-6 space-y-4">
            <form onSubmit={loginForm.handleSubmit(handleLogin)} className="space-y-4">
              <div className="space-y-2">
                <Label htmlFor="username">Kullanıcı Adı</Label>
                <Input
                  id="username"
                  type="text"
                  {...loginForm.register("username")}
                  placeholder="Kullanıcı adınızı girin"
                />
                {loginForm.formState.errors.username && (
                  <p className="text-sm text-red-500">{loginForm.formState.errors.username.message}</p>
                )}
              </div>
              
              <div className="space-y-2">
                <Label htmlFor="password">Şifre</Label>
                <Input
                  id="password"
                  type="password"
                  {...loginForm.register("password")}
                  placeholder="Şifrenizi girin"
                />
                {loginForm.formState.errors.password && (
                  <p className="text-sm text-red-500">{loginForm.formState.errors.password.message}</p>
                )}
              </div>
              
              <Button
                type="submit"
                className="w-full"
                disabled={loginMutation.isPending}
              >
                {loginMutation.isPending ? "Giriş yapılıyor..." : "Yönetim Paneline Giriş Yap"}
              </Button>

              <div className="pt-2 text-center text-sm text-muted-foreground">
                <p>Sadece yetkili yönetici kullanıcıları giriş yapabilir.</p>
              </div>
            </form>
          </CardContent>
        </Card>
      </div>
      
      {/* Hero column */}
      <div className="md:w-1/2 bg-gradient-to-br from-primary to-primary-foreground p-8 flex items-center justify-center">
        <div className="max-w-lg text-center text-white">
          <h1 className="text-4xl font-bold font-display mb-6">
            Jouska Cafe & Restaurant
          </h1>
          <p className="text-xl mb-8">
            Yönetim paneliyle menünüzü kolayca yönetin, özel kategori ve ürünler oluşturun.
          </p>
          <div className="space-y-4 text-left">
            <div className="flex items-center">
              <div className="bg-white/20 p-2 rounded-full mr-4">
                <svg xmlns="http://www.w3.org/2000/svg" className="h-6 w-6" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                  <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M5 13l4 4L19 7" />
                </svg>
              </div>
              <span>Menü öğelerini düzenleyin ve güncelleyin</span>
            </div>
            <div className="flex items-center">
              <div className="bg-white/20 p-2 rounded-full mr-4">
                <svg xmlns="http://www.w3.org/2000/svg" className="h-6 w-6" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                  <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M5 13l4 4L19 7" />
                </svg>
              </div>
              <span>Özel menüler ve kampanyalar oluşturun</span>
            </div>
            <div className="flex items-center">
              <div className="bg-white/20 p-2 rounded-full mr-4">
                <svg xmlns="http://www.w3.org/2000/svg" className="h-6 w-6" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                  <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M5 13l4 4L19 7" />
                </svg>
              </div>
              <span>Müşterileriniz için menü QR kodu oluşturun</span>
            </div>
          </div>
        </div>
      </div>
    </div>
  );
};

export default AuthPage;