import React, { useState } from "react";
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs";
import { useCategories } from "@/hooks/use-categories";
import { useMenuItems } from "@/hooks/use-menu-items";
import { useQuery } from "@tanstack/react-query";
import { Restaurant } from "@shared/schema";
import { Skeleton } from "@/components/ui/skeleton";
import { useToast } from "@/hooks/use-toast";
import { CategoryManagement } from "@/components/admin/category-management";
import { MenuItemManagement } from "@/components/admin/menu-item-management";
import { RestaurantSettings } from "@/components/admin/restaurant-settings";
import QRCodeGenerator from "@/components/admin/qr-code-generator";
import ThemeCustomizer from "@/components/admin/theme-customizer";

import { ImageImporter } from "@/components/admin/image-importer";
import { ChatbotSettings } from "@/components/admin/chatbot-settings";
import { SeoSettings } from "@/components/admin/seo-settings";
import { PaymentSettings } from "@/components/admin/payment-settings";
import { OrderManagement } from "@/components/admin/order-management";
import CustomerManagement from "@/components/admin/customer-management";
import CouponManagement from "@/components/admin/coupon-management";
import AnalyticsDashboard from "@/components/admin/analytics-dashboard";
import ReviewManagement from "@/components/admin/review-management";
import { useAuth } from "@/hooks/use-auth";
import { 
  BarChart3, 
  CreditCard,
  ImageIcon,
  ShoppingCart, 
  LayoutDashboard, 
  LogOut, 
  MessageCircle,
  Paintbrush, 
  QrCode, 
  Search,
  Settings, 
  Store, 
  User,
  Users,
  Gift,
  Star
} from "lucide-react";

export default function AdminPage() {
  const { toast } = useToast();
  const { user, logoutMutation } = useAuth();
  
  // Fetch data
  const { data: categories, isLoading: isLoadingCategories } = useCategories();
  const { data: menuItems, isLoading: isLoadingMenuItems } = useMenuItems();
  const restaurantQuery = useQuery<Restaurant>({ 
    queryKey: ['/api/restaurant'] 
  });
  
  const isLoading = isLoadingCategories || isLoadingMenuItems || restaurantQuery.isLoading;
  
  // Çıkış yap
  const handleLogout = async () => {
    try {
      await logoutMutation.mutateAsync();
      window.location.href = "/";
    } catch (error) {
      console.error("Çıkış hatası:", error);
      toast({
        title: "Çıkış hatası",
        description: "Oturumu kapatırken bir hata oluştu",
        variant: "destructive"
      });
    }
  };
  
  return (
    <div className="min-h-screen bg-gradient-to-b from-gray-50 to-gray-100">
      <header className="bg-primary/90 text-white shadow-md">
        <div className="max-w-7xl mx-auto py-5 px-4 sm:px-6 lg:px-8 flex justify-between items-center">
          <div className="flex items-center space-x-3">
            <div className="bg-accent/20 p-2 rounded-full">
              <svg xmlns="http://www.w3.org/2000/svg" className="h-7 w-7 text-accent" viewBox="0 0 24 24" fill="none" stroke="currentColor" strokeWidth="2" strokeLinecap="round" strokeLinejoin="round">
                <path d="M3 3h18v18H3z"></path>
                <path d="M8 12h8"></path>
                <path d="M12 8v8"></path>
              </svg>
            </div>
            <h1 className="text-3xl font-bold font-display">
              <span className="text-white">QR Menü</span>
              <span className="text-accent ml-2">Yönetim Paneli</span>
            </h1>
          </div>
          <div className="flex items-center space-x-4">
            <div className="hidden md:flex items-center mr-2 text-sm">
              <User className="h-4 w-4 mr-1" />
              <span>{user?.username}</span>
            </div>
            <button 
              onClick={() => window.location.href = "/"}
              className="px-4 py-2 text-sm text-primary-foreground bg-accent/20 hover:bg-accent/30 rounded-md transition flex items-center"
            >
              <svg xmlns="http://www.w3.org/2000/svg" className="h-4 w-4 mr-1" viewBox="0 0 24 24" fill="none" stroke="currentColor" strokeWidth="2" strokeLinecap="round" strokeLinejoin="round">
                <path d="M3 9l9-7 9 7v11a2 2 0 0 1-2 2H5a2 2 0 0 1-2-2z"></path>
                <polyline points="9 22 9 12 15 12 15 22"></polyline>
              </svg>
              Menü Görünümü
            </button>
            <button 
              onClick={handleLogout}
              className="px-4 py-2 text-sm bg-white/10 hover:bg-white/20 rounded-md transition flex items-center"
            >
              <LogOut className="h-4 w-4 mr-1" />
              Çıkış Yap
            </button>
          </div>
        </div>
      </header>
      
      <main className="max-w-7xl mx-auto py-8 sm:px-6 lg:px-8">
        {isLoading ? (
          <div className="p-6 space-y-4 bg-white rounded-lg shadow-sm">
            <Skeleton className="h-8 w-64" />
            <Skeleton className="h-64 w-full" />
          </div>
        ) : (
          <Tabs defaultValue="menu-items" className="w-full">
            <div className="grid grid-cols-1 lg:grid-cols-4 gap-6">
              {/* Sol Sidebar Menü */}
              <div className="lg:col-span-1">
                <div className="bg-white rounded-lg shadow-sm p-4 space-y-2 sticky top-6">
                  <h3 className="font-semibold text-gray-700 mb-4 text-sm uppercase tracking-wide">Admin Menü</h3>
                  
                  <TabsList className="flex flex-col h-auto bg-transparent p-0 space-y-1">
                    <TabsTrigger value="menu-items" className="w-full justify-start bg-transparent border-0 shadow-none hover:bg-gray-50 data-[state=active]:bg-primary data-[state=active]:text-white py-3 px-4">
                      <LayoutDashboard className="h-4 w-4 mr-3" />
                      <span>Menü Öğeleri</span>
                    </TabsTrigger>
                    <TabsTrigger value="categories" className="w-full justify-start bg-transparent border-0 shadow-none hover:bg-gray-50 data-[state=active]:bg-primary data-[state=active]:text-white py-3 px-4">
                      <Store className="h-4 w-4 mr-3" />
                      <span>Kategoriler</span>
                    </TabsTrigger>
                    <TabsTrigger value="orders" className="w-full justify-start bg-transparent border-0 shadow-none hover:bg-gray-50 data-[state=active]:bg-primary data-[state=active]:text-white py-3 px-4">
                      <ShoppingCart className="h-4 w-4 mr-3" />
                      <span>Sipariş Yönetimi</span>
                    </TabsTrigger>
                    <TabsTrigger value="restaurant" className="w-full justify-start bg-transparent border-0 shadow-none hover:bg-gray-50 data-[state=active]:bg-primary data-[state=active]:text-white py-3 px-4">
                      <Settings className="h-4 w-4 mr-3" />
                      <span>Restoran Bilgileri</span>
                    </TabsTrigger>
                    <TabsTrigger value="payments" className="w-full justify-start bg-transparent border-0 shadow-none hover:bg-gray-50 data-[state=active]:bg-primary data-[state=active]:text-white py-3 px-4">
                      <CreditCard className="h-4 w-4 mr-3" />
                      <span>Ödeme Sistemi</span>
                    </TabsTrigger>
                    <TabsTrigger value="qrcode" className="w-full justify-start bg-transparent border-0 shadow-none hover:bg-gray-50 data-[state=active]:bg-primary data-[state=active]:text-white py-3 px-4">
                      <QrCode className="h-4 w-4 mr-3" />
                      <span>QR Kod Üretici</span>
                    </TabsTrigger>
                    <TabsTrigger value="theme" className="w-full justify-start bg-transparent border-0 shadow-none hover:bg-gray-50 data-[state=active]:bg-primary data-[state=active]:text-white py-3 px-4">
                      <Paintbrush className="h-4 w-4 mr-3" />
                      <span>Tema Özelleştirme</span>
                    </TabsTrigger>
                    <TabsTrigger value="chatbot" className="w-full justify-start bg-transparent border-0 shadow-none hover:bg-gray-50 data-[state=active]:bg-primary data-[state=active]:text-white py-3 px-4">
                      <MessageCircle className="h-4 w-4 mr-3" />
                      <span>Chatbot Ayarları</span>
                    </TabsTrigger>
                    <TabsTrigger value="images" className="w-full justify-start bg-transparent border-0 shadow-none hover:bg-gray-50 data-[state=active]:bg-primary data-[state=active]:text-white py-3 px-4">
                      <ImageIcon className="h-4 w-4 mr-3" />
                      <span>Görsel Yönetimi</span>
                    </TabsTrigger>
                    <TabsTrigger value="seo" className="w-full justify-start bg-transparent border-0 shadow-none hover:bg-gray-50 data-[state=active]:bg-primary data-[state=active]:text-white py-3 px-4">
                      <Search className="h-4 w-4 mr-3" />
                      <span>SEO Yönetimi</span>
                    </TabsTrigger>
                    <TabsTrigger value="customers" className="w-full justify-start bg-transparent border-0 shadow-none hover:bg-gray-50 data-[state=active]:bg-primary data-[state=active]:text-white py-3 px-4">
                      <Users className="h-4 w-4 mr-3" />
                      <span>Müşteri Yönetimi</span>
                    </TabsTrigger>
                    <TabsTrigger value="coupons" className="w-full justify-start bg-transparent border-0 shadow-none hover:bg-gray-50 data-[state=active]:bg-primary data-[state=active]:text-white py-3 px-4">
                      <Gift className="h-4 w-4 mr-3" />
                      <span>Kupon Sistemi</span>
                    </TabsTrigger>
                    <TabsTrigger value="reviews" className="w-full justify-start bg-transparent border-0 shadow-none hover:bg-gray-50 data-[state=active]:bg-primary data-[state=active]:text-white py-3 px-4">
                      <Star className="h-4 w-4 mr-3" />
                      <span>Değerlendirmeler</span>
                    </TabsTrigger>
                    <TabsTrigger value="analytics" className="w-full justify-start bg-transparent border-0 shadow-none hover:bg-gray-50 data-[state=active]:bg-primary data-[state=active]:text-white py-3 px-4">
                      <BarChart3 className="h-4 w-4 mr-3" />
                      <span>Analytics Dashboard</span>
                    </TabsTrigger>
                  </TabsList>
                </div>
              </div>

              {/* Sağ İçerik Alanı */}
              <div className="lg:col-span-3">
                <TabsContent value="menu-items" className="bg-white p-6 rounded-lg shadow-sm">
                  <MenuItemManagement 
                    menuItems={menuItems || []} 
                    categories={categories || []} 
                  />
                </TabsContent>
                
                <TabsContent value="categories" className="bg-white p-6 rounded-lg shadow-sm">
                  <CategoryManagement categories={categories || []} />
                </TabsContent>
                
                <TabsContent value="orders" className="bg-white p-6 rounded-lg shadow-sm">
                  <OrderManagement />
                </TabsContent>
                
                <TabsContent value="restaurant" className="bg-white p-6 rounded-lg shadow-sm">
                  <RestaurantSettings restaurantInfo={restaurantQuery.data} />
                </TabsContent>

                <TabsContent value="payments" className="bg-white p-6 rounded-lg shadow-sm">
                  <PaymentSettings restaurantInfo={restaurantQuery.data} />
                </TabsContent>
                
                <TabsContent value="qrcode" className="bg-white p-6 rounded-lg shadow-sm">
                  <QRCodeGenerator />
                </TabsContent>
                
                <TabsContent value="theme" className="bg-white p-6 rounded-lg shadow-sm">
                  <ThemeCustomizer />
                </TabsContent>
                
                <TabsContent value="chatbot" className="bg-white p-6 rounded-lg shadow-sm">
                  <ChatbotSettings />
                </TabsContent>
                
                <TabsContent value="images" className="bg-white p-6 rounded-lg shadow-sm">
                  <ImageImporter />
                </TabsContent>
                
                <TabsContent value="seo" className="bg-white p-6 rounded-lg shadow-sm">
                  <SeoSettings />
                </TabsContent>
                
                <TabsContent value="customers" className="bg-white p-6 rounded-lg shadow-sm">
                  <CustomerManagement />
                </TabsContent>
                
                <TabsContent value="coupons" className="bg-white p-6 rounded-lg shadow-sm">
                  <CouponManagement />
                </TabsContent>
                
                <TabsContent value="reviews" className="bg-white p-6 rounded-lg shadow-sm">
                  <ReviewManagement />
                </TabsContent>
                
                <TabsContent value="analytics" className="bg-white p-6 rounded-lg shadow-sm">
                  <AnalyticsDashboard />
                </TabsContent>
              </div>
            </div>
          </Tabs>
        )}
      </main>
      
      <footer className="bg-white py-4 mt-12 border-t border-gray-200">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
          <div className="text-center text-sm text-gray-500">
            <p>&copy; {new Date().getFullYear()} Jouska Cafe & Restaurant - QR Menü Yönetim Sistemi</p>
          </div>
        </div>
      </footer>
    </div>
  );
}