// Basic service worker cache management for offline usage

export const MENU_CACHE_KEY = 'jouska-menu-cache';
export const CATEGORIES_CACHE_KEY = 'jouska-categories-cache';
export const RESTAURANT_CACHE_KEY = 'jouska-restaurant-cache';
export const LAST_UPDATED_KEY = 'jouska-data-last-updated';

// Cache menu items with timestamp
export const cacheMenuItems = (menuItems: any[]) => {
  try {
    localStorage.setItem(MENU_CACHE_KEY, JSON.stringify(menuItems));
    updateLastUpdated();
    return true;
  } catch (error) {
    console.error('Error caching menu items:', error);
    return false;
  }
};

// Cache categories with timestamp
export const cacheCategories = (categories: any[]) => {
  try {
    localStorage.setItem(CATEGORIES_CACHE_KEY, JSON.stringify(categories));
    updateLastUpdated();
    return true;
  } catch (error) {
    console.error('Error caching categories:', error);
    return false;
  }
};

// Cache restaurant info with timestamp
export const cacheRestaurantInfo = (restaurantInfo: any) => {
  try {
    localStorage.setItem(RESTAURANT_CACHE_KEY, JSON.stringify(restaurantInfo));
    updateLastUpdated();
    return true;
  } catch (error) {
    console.error('Error caching restaurant info:', error);
    return false;
  }
};

// Get cached menu items
export const getCachedMenuItems = () => {
  try {
    const cachedItems = localStorage.getItem(MENU_CACHE_KEY);
    return cachedItems ? JSON.parse(cachedItems) : null;
  } catch (error) {
    console.error('Error getting cached menu items:', error);
    return null;
  }
};

// Get cached categories
export const getCachedCategories = () => {
  try {
    const cachedCategories = localStorage.getItem(CATEGORIES_CACHE_KEY);
    return cachedCategories ? JSON.parse(cachedCategories) : null;
  } catch (error) {
    console.error('Error getting cached categories:', error);
    return null;
  }
};

// Get cached restaurant info
export const getCachedRestaurantInfo = () => {
  try {
    const cachedInfo = localStorage.getItem(RESTAURANT_CACHE_KEY);
    return cachedInfo ? JSON.parse(cachedInfo) : null;
  } catch (error) {
    console.error('Error getting cached restaurant info:', error);
    return null;
  }
};

// Update the last updated timestamp
const updateLastUpdated = () => {
  try {
    localStorage.setItem(LAST_UPDATED_KEY, new Date().toISOString());
  } catch (error) {
    console.error('Error updating last updated timestamp:', error);
  }
};

// Get the last updated timestamp
export const getLastUpdated = () => {
  try {
    const lastUpdated = localStorage.getItem(LAST_UPDATED_KEY);
    return lastUpdated || null;
  } catch (error) {
    console.error('Error getting last updated timestamp:', error);
    return null;
  }
};

// Check if data is cached
export const isDataCached = () => {
  return !!(getCachedMenuItems() && getCachedCategories() && getCachedRestaurantInfo());
};

// Calculate cache age in minutes
export const getCacheAge = () => {
  const lastUpdated = getLastUpdated();
  if (!lastUpdated) return null;
  
  const lastUpdatedDate = new Date(lastUpdated);
  const now = new Date();
  const diffMs = now.getTime() - lastUpdatedDate.getTime();
  const diffMinutes = Math.floor(diffMs / 60000);
  
  return diffMinutes;
};

// Check if cache is stale (older than 24 hours)
export const isCacheStale = () => {
  const cacheAge = getCacheAge();
  if (cacheAge === null) return true;
  
  // Consider cache stale after 24 hours (1440 minutes)
  return cacheAge > 1440;
};

// Clear all cached data
export const clearCachedData = () => {
  try {
    localStorage.removeItem(MENU_CACHE_KEY);
    localStorage.removeItem(CATEGORIES_CACHE_KEY);
    localStorage.removeItem(RESTAURANT_CACHE_KEY);
    localStorage.removeItem(LAST_UPDATED_KEY);
    return true;
  } catch (error) {
    console.error('Error clearing cached data:', error);
    return false;
  }
};