import { useQuery, useMutation } from "@tanstack/react-query";
import { queryClient, apiRequest } from "@/lib/queryClient";
import { useToast } from "./use-toast";
import { ThemeSettings } from "@shared/schema";

// Tema bilgilerini getir
export function useThemeSettings() {
  return useQuery<ThemeSettings | undefined>({
    queryKey: ['/api/themes/active'],
    refetchOnWindowFocus: false
  });
}

// Aktif temayı getir
export function useActiveTheme() {
  return useQuery<ThemeSettings>({
    queryKey: ['/api/themes/active']
  });
}

// Tüm temaları getir
export function useAllThemes() {
  return useQuery<ThemeSettings[]>({
    queryKey: ['/api/themes'],
    refetchOnWindowFocus: false
  });
}

// Tema oluştur
export function useCreateTheme() {
  const { toast } = useToast();
  
  return useMutation({
    mutationFn: async (themeData: any) => {
      const res = await apiRequest("/api/themes", {
        method: "POST",
        body: JSON.stringify(themeData)
      });
      return await res.json();
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ['/api/themes'] });
      queryClient.invalidateQueries({ queryKey: ['/api/themes/active'] });
      toast({
        title: "Tema oluşturuldu",
        description: "Yeni tema başarıyla oluşturuldu.",
      });
    },
    onError: (error: any) => {
      toast({
        title: "Tema oluşturulamadı",
        description: error.message || "Tema oluşturulurken bir hata oluştu.",
        variant: "destructive"
      });
    }
  });
}

// Tema güncelle
export function useUpdateTheme() {
  const { toast } = useToast();
  
  return useMutation({
    mutationFn: async ({ id, data }: { id: number; data: any }) => {
      const res = await apiRequest(`/api/themes/${id}`, {
        method: "PUT",
        body: JSON.stringify(data)
      });
      return await res.json();
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ['/api/themes'] });
      queryClient.invalidateQueries({ queryKey: ['/api/themes/active'] });
      toast({
        title: "Tema güncellendi",
        description: "Tema başarıyla güncellendi.",
      });
    },
    onError: (error: any) => {
      toast({
        title: "Tema güncellenemedi",
        description: error.message || "Tema güncellenirken bir hata oluştu.",
        variant: "destructive"
      });
    }
  });
}

// Temayı aktifleştir
export function useActivateTheme() {
  const { toast } = useToast();
  
  return useMutation({
    mutationFn: async (id: number) => {
      const res = await apiRequest(`/api/themes/${id}/activate`, {
        method: "PUT"
      });
      return await res.json();
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ['/api/themes'] });
      queryClient.invalidateQueries({ queryKey: ['/api/themes/active'] });
      toast({
        title: "Tema aktifleştirildi",
        description: "Tema başarıyla aktifleştirildi.",
      });
    },
    onError: (error: any) => {
      toast({
        title: "Tema aktifleştirilemedi",
        description: error.message || "Tema aktifleştirilirken bir hata oluştu.",
        variant: "destructive"
      });
    }
  });
}

// Tema sil
export function useDeleteTheme() {
  const { toast } = useToast();
  
  return useMutation({
    mutationFn: async (id: number) => {
      await apiRequest(`/api/themes/${id}`, {
        method: "DELETE"
      });
      return true;
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ['/api/themes'] });
      toast({
        title: "Tema silindi",
        description: "Tema başarıyla silindi.",
      });
    },
    onError: (error: any) => {
      toast({
        title: "Tema silinemedi",
        description: error.message || "Tema silinirken bir hata oluştu.",
        variant: "destructive"
      });
    }
  });
}