import React, { useEffect, useRef } from "react";
import QRCodeLib from "qrcode";

interface QRCodeProps {
  value: string;
  size?: number;
  primaryColor?: string;
  backgroundColor?: string;
}

export const QRCode = ({ 
  value, 
  size = 200, 
  primaryColor = "#8B4513",
  backgroundColor = "#ffffff" 
}: QRCodeProps) => {
  const canvasRef = useRef<HTMLCanvasElement>(null);

  useEffect(() => {
    if (canvasRef.current) {
      QRCodeLib.toCanvas(
        canvasRef.current,
        value,
        {
          width: size,
          margin: 1,
          color: {
            dark: primaryColor,
            light: backgroundColor,
          },
          errorCorrectionLevel: 'H'
        },
        (error) => {
          if (error) console.error("Error generating QR code:", error);
        }
      );
    }
  }, [value, size, primaryColor, backgroundColor]);

  return <canvas ref={canvasRef} className="rounded-lg shadow-sm" />;
};

export const downloadQRCode = (canvasElement: HTMLCanvasElement | null, filename = "menu-qr-code.png") => {
  if (!canvasElement) return;
  
  try {
    const dataUrl = canvasElement.toDataURL("image/png");
    const link = document.createElement("a");
    link.download = filename;
    link.href = dataUrl;
    link.click();
  } catch (err) {
    console.error("Error downloading QR code:", err);
  }
};
