import React, { useState } from "react";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Textarea } from "@/components/ui/textarea";
import { RadioGroup, RadioGroupItem } from "@/components/ui/radio-group";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Separator } from "@/components/ui/separator";
import { Badge } from "@/components/ui/badge";
import { useCart } from "@/hooks/use-cart";
import { useMutation, useQuery } from "@tanstack/react-query";
import { apiRequest } from "@/lib/queryClient";
import { toast } from "@/hooks/use-toast";
import { CreditCard, Wallet, Banknote, Loader2, User, Phone, MessageSquare } from "lucide-react";

interface CheckoutFormProps {
  isOpen: boolean;
  onClose: () => void;
}

export function CheckoutForm({ isOpen, onClose }: CheckoutFormProps) {
  const { items, totalPrice, clearCart } = useCart();
  const [customerName, setCustomerName] = useState("");
  const [customerPhone, setCustomerPhone] = useState("");
  const [tableNumber, setTableNumber] = useState("");
  const [notes, setNotes] = useState("");
  const [paymentMethod, setPaymentMethod] = useState<"paytr" | "iyzico" | "cash">("cash");

  // Get restaurant payment settings
  const { data: restaurantInfo } = useQuery({
    queryKey: ["/api/restaurant"],
  });

  // Get available payment methods based on restaurant settings
  const getAvailablePaymentMethods = () => {
    const methods: Array<{value: "paytr" | "iyzico" | "cash", label: string, icon: any, description: string}> = [];
    
    if (restaurantInfo?.enableCashPayment) {
      methods.push({
        value: "cash",
        label: "Nakit Ödeme",
        icon: Banknote,
        description: "Teslimat sırasında nakit olarak öde"
      });
    }
    
    if (restaurantInfo?.enablePaytr) {
      methods.push({
        value: "paytr",
        label: "PayTR",
        icon: CreditCard,
        description: "Kredi kartı ile güvenli ödeme"
      });
    }
    
    if (restaurantInfo?.enableIyzico) {
      methods.push({
        value: "iyzico",
        label: "Iyzico",
        icon: Wallet,
        description: "Kredi kartı ve dijital cüzdan"
      });
    }
    
    return methods;
  };

  const availablePaymentMethods = getAvailablePaymentMethods();

  // Set default payment method to first available option
  React.useEffect(() => {
    if (availablePaymentMethods.length > 0 && !availablePaymentMethods.some(m => m.value === paymentMethod)) {
      setPaymentMethod(availablePaymentMethods[0].value);
    }
  }, [availablePaymentMethods, paymentMethod]);

  const createOrderMutation = useMutation({
    mutationFn: async (orderData: any) => {
      const response = await apiRequest("POST", "/api/orders", orderData);
      return response.json();
    },
    onSuccess: (data) => {
      if (data.success) {
        if (data.paymentUrl) {
          // Redirect to payment page
          window.open(data.paymentUrl, "_blank");
          toast({
            title: "Ödeme sayfasına yönlendiriliyorsunuz",
            description: "Ödeme işlemini tamamlamak için açılan sayfayı kullanın.",
          });
        } else {
          toast({
            title: "Sipariş alındı!",
            description: data.message || "Siparişiniz başarıyla alındı.",
          });
          clearCart();
          onClose();
        }
      }
    },
    onError: (error: any) => {
      toast({
        title: "Sipariş hatası",
        description: error.message || "Sipariş oluşturulamadı. Lütfen tekrar deneyin.",
        variant: "destructive",
      });
    },
  });

  const handleSubmit = (e: React.FormEvent) => {
    e.preventDefault();
    
    if (!customerName.trim()) {
      toast({
        title: "Ad Soyad gerekli",
        description: "Lütfen adınızı ve soyadınızı girin.",
        variant: "destructive",
      });
      return;
    }

    if (items.length === 0) {
      toast({
        title: "Sepet boş",
        description: "Sipariş vermek için sepetinize ürün eklemelisiniz.",
        variant: "destructive",
      });
      return;
    }

    const orderData = {
      customerName: customerName.trim(),
      customerPhone: customerPhone.trim() || undefined,
      tableNumber: tableNumber.trim() || undefined,
      totalAmount: totalPrice.toFixed(2),
      notes: notes.trim() || undefined,
      paymentMethod,
      items: items.map(item => ({
        menuItemId: item.menuItem.id,
        name: item.menuItem.name,
        price: item.menuItem.price,
        quantity: item.quantity,
        notes: item.notes,
      })),
    };

    createOrderMutation.mutate(orderData);
  };

  if (!isOpen) return null;

  return (
    <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50 p-4">
      <Card className="w-full max-w-2xl max-h-[90vh] overflow-y-auto">
        <CardHeader>
          <CardTitle className="flex items-center justify-between">
            <span className="flex items-center gap-2">
              <CreditCard className="w-6 h-6 text-orange-500" />
              Sipariş Tamamla
            </span>
            <Button variant="ghost" onClick={onClose} className="p-2">
              ✕
            </Button>
          </CardTitle>
        </CardHeader>
        
        <CardContent className="space-y-6">
          {/* Order Summary */}
          <div>
            <h3 className="font-semibold mb-3">Sipariş Özeti</h3>
            <div className="space-y-2 mb-4">
              {items.map((item, index) => (
                <div key={index} className="flex justify-between items-center py-2 border-b">
                  <div>
                    <span className="font-medium">{item.quantity}x {item.menuItem.name}</span>
                    {item.notes && (
                      <p className="text-sm text-gray-600">Not: {item.notes}</p>
                    )}
                  </div>
                  <span className="font-medium">
                    ₺{(parseFloat(item.menuItem.price.replace(' ₺', '')) * item.quantity).toFixed(2)}
                  </span>
                </div>
              ))}
            </div>
            <div className="flex justify-between items-center text-lg font-bold">
              <span>Toplam:</span>
              <span>₺{totalPrice.toFixed(2)}</span>
            </div>
          </div>

          <Separator />

          {/* Customer Information */}
          <form onSubmit={handleSubmit} className="space-y-4">
            <div>
              <h3 className="font-semibold mb-3 flex items-center gap-2">
                <User className="w-5 h-5" />
                Müşteri Bilgileri
              </h3>
              <div className="grid gap-4">
                <div>
                  <Label htmlFor="customerName">Ad Soyad *</Label>
                  <Input
                    id="customerName"
                    value={customerName}
                    onChange={(e) => setCustomerName(e.target.value)}
                    placeholder="Adınız ve soyadınız"
                    required
                  />
                </div>
                
                <div>
                  <Label htmlFor="customerPhone">Telefon</Label>
                  <Input
                    id="customerPhone"
                    type="tel"
                    value={customerPhone}
                    onChange={(e) => setCustomerPhone(e.target.value)}
                    placeholder="5XX XXX XX XX"
                  />
                </div>
                
                <div>
                  <Label htmlFor="tableNumber">Masa Numarası</Label>
                  <Input
                    id="tableNumber"
                    value={tableNumber}
                    onChange={(e) => setTableNumber(e.target.value)}
                    placeholder="Masa numaranız (varsa)"
                  />
                </div>
                
                <div>
                  <Label htmlFor="notes">Özel Notlar</Label>
                  <Textarea
                    id="notes"
                    value={notes}
                    onChange={(e) => setNotes(e.target.value)}
                    placeholder="Siparişiniz hakkında özel talepleriniz..."
                    rows={3}
                  />
                </div>
              </div>
            </div>

            <Separator />

            {/* Payment Method */}
            {restaurantInfo?.enablePayments && availablePaymentMethods.length > 0 ? (
              <div>
                <h3 className="font-semibold mb-3">Ödeme Yöntemi</h3>
                <RadioGroup value={paymentMethod} onValueChange={(value) => setPaymentMethod(value as any)}>
                  <div className="grid gap-3">
                    {availablePaymentMethods.map((method) => {
                      const IconComponent = method.icon;
                      return (
                        <div key={method.value} className="flex items-center space-x-2 p-3 border rounded-lg hover:bg-gray-50">
                          <RadioGroupItem value={method.value} id={method.value} />
                          <Label htmlFor={method.value} className="flex items-center gap-2 cursor-pointer flex-1">
                            <IconComponent className={`w-5 h-5 ${
                              method.value === 'cash' ? 'text-green-600' :
                              method.value === 'paytr' ? 'text-blue-600' :
                              'text-purple-600'
                            }`} />
                            <div>
                              <div className="font-medium flex items-center gap-2">
                                {method.label}
                                {method.value !== 'cash' && <Badge variant="secondary" className="text-xs">Güvenli</Badge>}
                              </div>
                              <div className="text-sm text-gray-600">{method.description}</div>
                            </div>
                          </Label>
                        </div>
                      );
                    })}
                  </div>
                </RadioGroup>
              </div>
            ) : (
              <div className="bg-yellow-50 border border-yellow-200 rounded-lg p-4">
                <div className="flex items-center gap-2 mb-2">
                  <CreditCard className="w-5 h-5 text-yellow-600" />
                  <h4 className="text-sm font-medium text-yellow-800">Ödeme Sistemi Kapalı</h4>
                </div>
                <p className="text-sm text-yellow-700">
                  Şu anda online sipariş ve ödeme sistemi devre dışıdır. 
                  Sipariş vermek için lütfen doğrudan restoranla iletişime geçin.
                </p>
              </div>
            )}

            <Separator />

            {/* Submit Button */}
            {restaurantInfo?.enablePayments && availablePaymentMethods.length > 0 ? (
              <div className="flex gap-3">
                <Button
                  type="button"
                  variant="outline"
                  onClick={onClose}
                  className="flex-1"
                  disabled={createOrderMutation.isPending}
                >
                  İptal
                </Button>
                <Button
                  type="submit"
                  className="flex-1 bg-orange-500 hover:bg-orange-600"
                  disabled={createOrderMutation.isPending}
                >
                  {createOrderMutation.isPending ? (
                    <>
                      <Loader2 className="w-4 h-4 mr-2 animate-spin" />
                      İşleniyor...
                    </>
                  ) : (
                    <>
                      {paymentMethod === 'cash' ? 'Siparişi Tamamla' : 'Ödemeye Geç'}
                    </>
                  )}
                </Button>
              </div>
            ) : (
              <div className="flex gap-3">
                <Button
                  type="button"
                  variant="outline"
                  onClick={onClose}
                  className="flex-1"
                >
                  Kapat
                </Button>
              </div>
            )}
          </form>
        </CardContent>
      </Card>
    </div>
  );
}