import React, { useRef } from "react";
import { Dialog, DialogContent, DialogHeader, DialogTitle, DialogClose } from "@/components/ui/dialog";
import { Button } from "@/components/ui/button";
import { QRCode, downloadQRCode } from "@/components/ui/qr-code";
import { X, Download } from "lucide-react";

interface QRModalProps {
  open: boolean;
  onOpenChange: (open: boolean) => void;
  url: string;
}

export const QRModal = ({ open, onOpenChange, url }: QRModalProps) => {
  const qrCanvasRef = useRef<HTMLCanvasElement | null>(null);
  
  const handleDownload = () => {
    downloadQRCode(qrCanvasRef.current);
  };
  
  const setCanvasRef = (canvas: HTMLCanvasElement | null) => {
    if (canvas) {
      qrCanvasRef.current = canvas;
    }
  };
  
  return (
    <Dialog open={open} onOpenChange={onOpenChange}>
      <DialogContent className="sm:max-w-md border-accent/20 overflow-hidden p-0">
        <div className="bg-gradient-to-b from-primary/10 to-transparent pt-6 px-6">
          <DialogHeader className="mb-4">
            <DialogTitle className="font-display text-2xl font-bold text-primary">
              QR Menü Kodu
            </DialogTitle>
            <DialogClose className="absolute right-4 top-4 rounded-full w-8 h-8 flex items-center justify-center border border-border hover:bg-muted text-neutral-dark hover:text-primary transition-colors">
              <X className="h-4 w-4" />
            </DialogClose>
          </DialogHeader>
        
          <div className="flex flex-col items-center">
            <div className="bg-white p-6 rounded-xl shadow-md mb-6 relative">
              <div className="absolute inset-0 bg-gradient-to-br from-accent/5 to-primary/5 rounded-xl"></div>
              <div className="relative">
                <QRCode 
                  value={url} 
                  size={220} 
                  primaryColor="hsl(var(--primary))" 
                  backgroundColor="#ffffff"
                  // @ts-ignore - this is a custom ref implementation
                  ref={setCanvasRef}
                />
                <div className="absolute inset-0 flex items-center justify-center opacity-20">
                  <span className="font-display text-primary text-lg">Gusto</span>
                </div>
              </div>
            </div>
            
            <div className="bg-secondary/20 rounded-lg p-4 mb-6 text-center w-full">
              <p className="text-sm text-gray-600 mb-2">
                Dijital menümüze erişmek için QR kodu telefonunuzun kamerasıyla tarayın.
              </p>
              <p className="text-xs text-gray-500">
                Link: <span className="text-primary font-medium">{url}</span>
              </p>
            </div>
            
            <div className="mb-6 w-full space-y-2">
              <Button 
                onClick={handleDownload} 
                className="w-full bg-accent hover:bg-accent/90 text-white py-2 rounded-lg text-sm font-medium flex items-center justify-center transition duration-200 gap-2"
              >
                <Download className="h-4 w-4" />
                QR Kodu İndir
              </Button>
              
              <Button
                variant="outline"
                className="w-full border-primary/20 text-primary hover:bg-primary/5"
                onClick={() => {
                  // Copy URL to clipboard
                  navigator.clipboard.writeText(url);
                  // You could add a toast notification here
                }}
              >
                <svg xmlns="http://www.w3.org/2000/svg" className="h-4 w-4 mr-2" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                  <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M8 5H6a2 2 0 00-2 2v12a2 2 0 002 2h10a2 2 0 002-2v-1M8 5a2 2 0 002 2h2a2 2 0 002-2M8 5a2 2 0 012-2h2a2 2 0 012 2m0 0h2a2 2 0 012 2v3m2 4H10m0 0l3-3m-3 3l3 3" />
                </svg>
                Bağlantıyı Kopyala
              </Button>
            </div>
          </div>
        </div>
      </DialogContent>
    </Dialog>
  );
};
