import React, { useState } from "react";
import { Dialog, DialogContent, DialogHeader, DialogTitle, DialogClose, DialogDescription } from "@/components/ui/dialog";
import { Badge } from "@/components/ui/badge";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Textarea } from "@/components/ui/textarea";
import { X, ChevronLeft, ChevronRight, Plus, Minus, ShoppingCart } from "lucide-react";
import { MenuItem } from "@shared/schema";
import { useCart } from "@/hooks/use-cart";
import { toast } from "@/hooks/use-toast";
import { useQuery } from "@tanstack/react-query";

interface MenuItemDetailModalProps {
  open: boolean;
  onOpenChange: (open: boolean) => void;
  item?: MenuItem;
}

export const MenuItemDetailModal = ({ open, onOpenChange, item }: MenuItemDetailModalProps) => {
  const [quantity, setQuantity] = useState(1);
  const [notes, setNotes] = useState("");
  const [currentImageIndex, setCurrentImageIndex] = useState(0);
  const { addItem } = useCart();
  
  // Restaurant bilgisini çek
  const { data: restaurantInfo } = useQuery({
    queryKey: ['/api/restaurant'],
  });

  if (!item) {
    console.error("No item provided to MenuItemDetailModal");
    return null;
  }
  
  // Görselleri işleyip tek bir dizi haline getiren fonksiyon
  const getImages = () => {
    const result: string[] = [];
    
    // Ana görseli ekle (eğer varsa) - backend'den 'image' field'ı geliyor
    if (item.image) {
      result.push(item.image);
    }
    
    // Ek görselleri ekle (eğer varsa) - backend'den 'images' field'ı geliyor
    if (typeof item.images === 'string' && item.images.trim() !== '') {
      const additionalUrls = item.images
        .split(',')
        .map(url => url.trim())
        .filter(url => url !== '');
        
      result.push(...additionalUrls);
    }
    
    return result;
  };
  
  // Tüm görselleri bir kereliğine hesapla ve saklayalım
  const allImages = React.useMemo(() => getImages(), [item.image, item.images]);

  // Debug bilgilerini kaldırıyoruz, gereksiz konsol günlüğü oluşturuyorlar
  
  const hasMultipleImages = allImages.length > 1;
  
  const nextImage = () => {
    setCurrentImageIndex((prev) => (prev + 1) % allImages.length);
  };

  const prevImage = () => {
    setCurrentImageIndex((prev) => (prev - 1 + allImages.length) % allImages.length);
  };

  return (
    <Dialog open={open} onOpenChange={onOpenChange}>
      <DialogContent className="sm:max-w-3xl overflow-y-auto max-h-[90vh] bg-card text-card-foreground p-0" closeButton={false}>
        {/* Adding a DialogTitle for accessibility */}
        <DialogTitle className="sr-only">{item.name} Details</DialogTitle>
        
        {/* Adding DialogDescription for accessibility */}
        <DialogDescription className="sr-only">
          A detailed view of the menu item with image gallery, price, availability, and description.
        </DialogDescription>
        
        {/* Hero section with image and overlay */}
        {allImages.length > 0 && (
          <div className="relative h-[300px] sm:h-[350px] md:h-[400px] lg:h-[500px] w-full">
            <div className="absolute inset-0 bg-gradient-to-t from-black/70 via-black/40 to-black/5 z-10"></div>
            
            {/* Current Image */}
            <div className="w-full h-full bg-black">
              <img 
                src={allImages[currentImageIndex]} 
                alt={`${item.name} - image ${currentImageIndex + 1}`}
                className="w-full h-full object-cover"
                loading="lazy"
                decoding="async"
                onError={(e) => {
                  // Yedek görsel - Jouska restoranı için standart bir yemek görseli
                  e.currentTarget.src = "https://images.unsplash.com/photo-1540189549336-e6e99c3679fe?w=500&auto=format&fit=crop&q=80";
                }}
              />
            </div>
            
            {/* Navigation arrows for multiple images */}
            {hasMultipleImages && (
              <>
                <button
                  onClick={prevImage}
                  className="absolute left-4 top-1/2 -translate-y-1/2 w-10 h-10 bg-black/50 hover:bg-black/70 rounded-full flex items-center justify-center text-white transition-colors z-20"
                  aria-label="Previous image"
                >
                  <ChevronLeft className="w-6 h-6" />
                </button>
                <button
                  onClick={nextImage}
                  className="absolute right-4 top-1/2 -translate-y-1/2 w-10 h-10 bg-black/50 hover:bg-black/70 rounded-full flex items-center justify-center text-white transition-colors z-20"
                  aria-label="Next image"
                >
                  <ChevronRight className="w-6 h-6" />
                </button>
              </>
            )}
            
            {/* Image counter */}
            {hasMultipleImages && (
              <div className="absolute bottom-4 right-4 bg-black/50 text-white px-3 py-1 rounded-full text-sm z-20">
                {currentImageIndex + 1} / {allImages.length}
              </div>
            )}
            
            {/* Title and price overlay */}
            <div className="absolute bottom-0 left-0 right-0 p-6 z-10">
              <DialogTitle className="font-display text-4xl font-bold text-white mb-3 leading-tight">{item.name}</DialogTitle>
              <div className="flex items-center gap-3 mb-4">
                {item.isCampaign && item.campaignPrice ? (
                  <div className="flex items-center gap-2">
                    <span className="bg-white/20 text-white/80 font-bold font-condensed px-2 py-1 rounded-full shadow-lg line-through text-sm">
                      ₺{item.price.replace(' ₺', '')}
                    </span>
                    <span className="bg-orange-600 text-white font-bold font-condensed px-3 py-1.5 rounded-full shadow-lg">
                      ₺{item.campaignPrice?.replace(' ₺', '') || ''}
                    </span>
                  </div>
                ) : (
                  <span className="bg-primary text-white font-bold font-condensed px-3 py-1.5 rounded-full shadow-lg">
                    ₺{item.price.replace(' ₺', '')}
                  </span>
                )}
              </div>
            </div>
            
            <DialogClose className="absolute right-4 top-4 rounded-full w-8 h-8 flex items-center justify-center bg-black/30 hover:bg-black/50 dark:bg-white/10 dark:hover:bg-white/20 text-white hover:text-white transition-colors z-20">
              <X className="h-4 w-4" />
            </DialogClose>
          </div>
        )}
        
        {/* If no image, show a simpler header */}
        {allImages.length === 0 && (
          <div className="relative bg-primary text-white p-6">
            <DialogTitle className="font-display text-3xl font-bold mb-2">{item.name}</DialogTitle>
            <div className="inline-block">
              {item.isCampaign && item.campaignPrice ? (
                <div className="flex items-center gap-2">
                  <span className="bg-white/20 text-white/80 font-bold font-condensed px-2 py-1 rounded-full shadow-md line-through text-sm">
                    ₺{item.price.replace(' ₺', '')}
                  </span>
                  <span className="bg-orange-600 text-white font-bold font-condensed px-3 py-1.5 rounded-full shadow-md">
                    ₺{item.campaignPrice?.replace(' ₺', '') || ''}
                  </span>
                </div>
              ) : (
                <span className="bg-primary text-white font-bold font-condensed px-3 py-1.5 rounded-full shadow-md">
                  ₺{item.price.replace(' ₺', '')}
                </span>
              )}
            </div>
            {/* No close button here in the no-image version, as we use the main one */}
          </div>
        )}
        
        <div className="p-6 pt-3">
          <div className="flex flex-col gap-6">
            {/* Availability status and tags */}
            <div className="flex flex-wrap items-center gap-3">
              <div className="flex items-center">
                <div className={`h-3 w-3 rounded-full mr-2 ${item.isActive ? 'bg-green-500' : 'bg-red-500'}`}></div>
                <span className={`text-sm font-medium ${item.isActive ? 'text-green-700' : 'text-red-700'}`}>
                  {item.isActive ? 'Sipariş Verilebilir' : 'Tükendi'}
                </span>
              </div>
              
              {item.isVegetarian && (
                <Badge variant="outline" className="bg-green-100 text-green-800 hover:bg-green-200 border-green-200">
                  <svg xmlns="http://www.w3.org/2000/svg" className="h-3.5 w-3.5 mr-1" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                    <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M5 13l4 4L19 7" />
                  </svg>
                  Vejetaryen
                </Badge>
              )}
              
              {item.isVegan && (
                <Badge variant="outline" className="bg-emerald-100 text-emerald-800 hover:bg-emerald-200 border-emerald-200">
                  <svg xmlns="http://www.w3.org/2000/svg" className="h-3.5 w-3.5 mr-1" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                    <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M5 13l4 4L19 7" />
                  </svg>
                  Vegan
                </Badge>
              )}
              
              {item.isGlutenFree && (
                <Badge variant="outline" className="bg-blue-100 text-blue-800 hover:bg-blue-200 border-blue-200">
                  <svg xmlns="http://www.w3.org/2000/svg" className="h-3.5 w-3.5 mr-1" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                    <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M9 12l2 2 4-4M7.835 4.697a3.42 3.42 0 001.946-.806 3.42 3.42 0 014.438 0 3.42 3.42 0 001.946.806 3.42 3.42 0 013.138 3.138 3.42 3.42 0 00.806 1.946 3.42 3.42 0 010 4.438 3.42 3.42 0 00-.806 1.946 3.42 3.42 0 01-3.138 3.138 3.42 3.42 0 00-1.946.806 3.42 3.42 0 01-4.438 0 3.42 3.42 0 00-1.946-.806 3.42 3.42 0 01-3.138-3.138 3.42 3.42 0 00-.806-1.946 3.42 3.42 0 010-4.438 3.42 3.42 0 00.806-1.946 3.42 3.42 0 013.138-3.138z" />
                  </svg>
                  Glutensiz
                </Badge>
              )}
              
              {item.isSpicy && (
                <Badge variant="outline" className="bg-red-100 text-red-800 hover:bg-red-200 border-red-200">
                  🌶️ Acılı
                </Badge>
              )}
              
              {item.isPopular && (
                <Badge variant="outline" className="bg-orange-100 text-orange-800 hover:bg-orange-200 border-orange-200">
                  <svg xmlns="http://www.w3.org/2000/svg" className="h-3.5 w-3.5 mr-1" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                    <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M13 10V3L4 14h7v7l9-11h-7z" />
                  </svg>
                  Popüler
                </Badge>
              )}
            </div>
            
            {/* Description */}
            {item.description && (
              <div>
                <h3 className="text-lg font-bold mb-2 font-display text-primary">Açıklama</h3>
                <p className="text-foreground leading-relaxed">{item.description}</p>
              </div>
            )}
            
            {/* Nutrition information - Only show if order system is enabled */}
            {restaurantInfo?.enableOrderSystem && (
              <div className="bg-orange-50 border border-orange-200 p-5 rounded-lg">
                <h3 className="text-lg font-bold mb-3 font-display text-orange-700 border-b border-orange-200 pb-2 flex items-center">
                  <svg xmlns="http://www.w3.org/2000/svg" className="h-5 w-5 mr-2" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                    <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M9 19v-6a2 2 0 00-2-2H5a2 2 0 00-2 2v6a2 2 0 002 2h2a2 2 0 002-2zm0 0V9a2 2 0 012-2h2a2 2 0 012 2v10m-6 0a2 2 0 002 2h2a2 2 0 002-2m0 0V5a2 2 0 012-2h2a2 2 0 012 2v14a2 2 0 01-2 2h-2a2 2 0 01-2-2z" />
                  </svg>
                  Besin Değerleri ve Bilgiler
                </h3>
                <p className="text-sm text-orange-600 mb-4">
                  Aşağıdaki değerler yaklaşıktır ve porsiyon büyüklüğüne göre değişebilir. Detaylı bilgi için personele danışabilirsiniz.
                </p>
                <div className="grid grid-cols-4 gap-0 text-sm border border-orange-200 rounded-md overflow-hidden bg-white">
                  <div className="p-3 text-center font-medium bg-orange-100 border-r border-orange-200 text-orange-800">Kalori</div>
                  <div className="p-3 text-center font-medium bg-orange-100 border-r border-orange-200 text-orange-800">Protein</div>
                  <div className="p-3 text-center font-medium bg-orange-100 border-r border-orange-200 text-orange-800">Karbonhidrat</div>
                  <div className="p-3 text-center font-medium bg-orange-100 text-orange-800">Yağ</div>
                  
                  <div className="p-3 text-center border-r border-orange-200">{item.name.includes("Toast") ? "350 kcal" : item.name.includes("Burger") ? "420 kcal" : item.name.includes("Kahve") ? "2 kcal" : "280 kcal"}</div>
                  <div className="p-3 text-center border-r border-orange-200">{item.name.includes("Kahve") ? "1g" : item.name.includes("Burger") ? "25g" : item.name.includes("Toast") ? "12g" : "15g"}</div>
                  <div className="p-3 text-center border-r border-orange-200">{item.name.includes("Kahve") || item.name.includes("Çay") ? "0g" : item.name.includes("Toast") ? "30g" : "25g"}</div>
                  <div className="p-3 text-center">{item.name.includes("Kahve") || item.name.includes("Çay") ? "0g" : item.name.includes("Toast") ? "15g" : "18g"}</div>
                </div>
              </div>
            )}
            
            {/* Information cards - Only show if order system is enabled */}
            {restaurantInfo?.enableOrderSystem && (
              <div className="grid grid-cols-2 gap-3 mt-4">
                <div className="bg-card p-3 rounded-md shadow-sm border border-border flex items-center">
                  <div className="h-8 w-8 rounded-full bg-accent/10 flex items-center justify-center mr-3">
                    <svg xmlns="http://www.w3.org/2000/svg" className="h-4 w-4 text-accent" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                      <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M13 16h-1v-4h-1m1-4h.01M21 12a9 9 0 11-18 0 9 9 0 0118 0z" />
                    </svg>
                  </div>
                  <div>
                    <div className="text-xs text-muted-foreground">Alerjenler</div>
                    <div className="text-sm font-medium">
                      {item.isVegetarian ? "Vejetaryen uygun" : "Talep üzerine"}
                    </div>
                  </div>
                </div>
                
                <div className="bg-card p-3 rounded-md shadow-sm border border-border flex items-center">
                  <div className="h-8 w-8 rounded-full bg-primary/10 flex items-center justify-center mr-3">
                    <svg xmlns="http://www.w3.org/2000/svg" className="h-4 w-4 text-primary" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                      <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M12 8v4l3 3m6-3a9 9 0 11-18 0 9 9 0 0118 0z" />
                    </svg>
                  </div>
                  <div>
                    <div className="text-xs text-muted-foreground">Hazırlanma Süresi</div>
                    <div className="text-sm font-medium">
                      {item.name.includes("Kahve") || item.name.includes("Çay") ? "3-5 dk" : item.name.includes("Salata") ? "10-15 dk" : "15-20 dk"}
                    </div>
                  </div>
                </div>
              </div>
            )}
          </div>
          
          {/* Recommendations - Only show if order system is enabled */}
          {restaurantInfo?.enableOrderSystem && (
            <div className="bg-primary/5 p-5 rounded-lg border border-primary/10">
              <h3 className="text-lg font-bold mb-3 font-display text-primary border-b border-primary/10 pb-2 flex items-center">
                <svg xmlns="http://www.w3.org/2000/svg" className="h-5 w-5 mr-2 text-accent" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                  <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M9.663 17h4.673M12 3v1m6.364 1.636l-.707.707M21 12h-1M4 12H3m3.343-5.657l-.707-.707m2.828 9.9a5 5 0 117.072 0l-.548.547A3.374 3.374 0 0014 18.469V19a2 2 0 11-4 0v-.531c0-.895-.356-1.754-.988-2.386l-.548-.547z" />
                  </svg>
                Öneriler
              </h3>
              <div className="text-sm text-foreground">
                <div className="flex flex-col gap-2">
                  <div className="flex items-center">
                    <svg xmlns="http://www.w3.org/2000/svg" className="h-4 w-4 mr-2 text-primary" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                      <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M9 12l2 2 4-4M7.835 4.697a3.42 3.42 0 001.946-.806 3.42 3.42 0 014.438 0 3.42 3.42 0 001.946.806 3.42 3.42 0 013.138 3.138 3.42 3.42 0 00.806 1.946 3.42 3.42 0 010 4.438 3.42 3.42 0 00-.806 1.946 3.42 3.42 0 01-3.138 3.138 3.42 3.42 0 00-1.946.806 3.42 3.42 0 01-4.438 0 3.42 3.42 0 00-1.946-.806 3.42 3.42 0 01-3.138-3.138 3.42 3.42 0 00-.806-1.946 3.42 3.42 0 010-4.438 3.42 3.42 0 00.806-1.946 3.42 3.42 0 013.138-3.138z" />
                    </svg>
                    <div>
                      Bu ürün şefimizin özel tarifiye hazırlanır.
                    </div>
                  </div>
                  <div className="flex items-center">
                    <svg xmlns="http://www.w3.org/2000/svg" className="h-4 w-4 mr-2 text-accent" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                      <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M5 13l4 4L19 7" />
                    </svg>
                    <div>
                      {item.name.includes("Burger") ? 
                       "Patates kızartması ile harika bir kombinasyon oluşturur." :
                       item.name.includes("Kahve") ?
                       "Tatlılarımızla mükemmel uyum sağlar." :
                       "Ekstra sos talep edebilirsiniz."}
                    </div>
                  </div>
                </div>
              </div>
            </div>
          )}
          
          {/* Add to Cart Section - Only show if order system is enabled */}
          {restaurantInfo?.enableOrderSystem && (
            <div className="border-t pt-6 mt-6">
              <h3 className="text-lg font-bold mb-4 font-display text-primary flex items-center">
                <ShoppingCart className="h-5 w-5 mr-2 text-orange-500" />
                Sepete Ekle
              </h3>
              
              <div className="space-y-4">
                {/* Quantity Selector */}
                <div className="flex items-center justify-between">
                  <span className="font-medium">Adet:</span>
                  <div className="flex items-center space-x-3">
                    <Button
                      variant="outline"
                      size="sm"
                      onClick={() => setQuantity(Math.max(1, quantity - 1))}
                      className="h-8 w-8 p-0"
                    >
                      <Minus className="h-4 w-4" />
                    </Button>
                    <span className="w-8 text-center font-semibold">{quantity}</span>
                    <Button
                      variant="outline"
                      size="sm"
                      onClick={() => setQuantity(quantity + 1)}
                      className="h-8 w-8 p-0"
                    >
                      <Plus className="h-4 w-4" />
                    </Button>
                  </div>
                </div>
                
                {/* Special Notes */}
                <div>
                  <label className="block text-sm font-medium mb-2">Özel Not (isteğe bağlı):</label>
                  <Textarea
                    placeholder="Özel taleplerinizi buraya yazabilirsiniz..."
                    value={notes}
                    onChange={(e) => setNotes(e.target.value)}
                    className="resize-none"
                    rows={3}
                  />
                </div>
                
                {/* Price and Add Button */}
                <div className="flex items-center justify-between pt-4 border-t">
                  <div className="text-2xl font-bold text-primary">
                    ₺{(parseFloat(item.price.replace(' ₺', '')) * quantity).toFixed(2)}
                  </div>
                  <Button
                    onClick={() => {
                      addItem(item, quantity, notes);
                      toast({
                        title: "Sepete eklendi",
                        description: `${quantity}x ${item.name} sepetinize eklendi.`,
                        variant: "default",
                      });
                      onOpenChange(false);
                    }}
                    className="bg-orange-500 hover:bg-orange-600 text-white px-6 py-3 text-lg flex items-center gap-2"
                  >
                    <Plus className="w-5 h-5" />
                    Sepete Ekle
                  </Button>
                </div>
              </div>
            </div>
          )}
        </div>
      </DialogContent>
    </Dialog>
  );
};