import React, { useState, useEffect, useRef } from "react";
import { Category } from "@shared/schema";
import { Link } from "wouter";

interface CategoryNavigationProps {
  categories: Category[];
  selectedCategory?: string | null;
  onCategorySelect?: (categorySlug: string | null) => void;
}

export const CategoryNavigation = ({ categories, selectedCategory, onCategorySelect }: CategoryNavigationProps) => {
  const [activeCategory, setActiveCategory] = useState<string>("");
  const [hasScrolled, setHasScrolled] = useState<boolean>(false);
  const [canScrollLeft, setCanScrollLeft] = useState<boolean>(false);
  const [canScrollRight, setCanScrollRight] = useState<boolean>(true);
  
  const scrollContainerRef = useRef<HTMLDivElement>(null);
  
  // Check scroll possibilities
  const checkScroll = () => {
    if (scrollContainerRef.current) {
      const { scrollLeft, scrollWidth, clientWidth } = scrollContainerRef.current;
      setCanScrollLeft(scrollLeft > 0);
      setCanScrollRight(scrollLeft < scrollWidth - clientWidth - 5); // 5px buffer for precision
      
      // Debug scroll values
      console.log("Scroll check:", { 
        scrollLeft, 
        scrollWidth, 
        clientWidth,
        canScroll: scrollWidth > clientWidth,
        difference: scrollWidth - clientWidth
      });
    }
  };
  
  // Set initial active category or update when scrolling
  useEffect(() => {
    if (categories.length > 0 && !activeCategory) {
      setActiveCategory(categories[0].slug);
    }
    
    const handleScroll = () => {
      const scrollPosition = window.scrollY;
      
      // After user has scrolled a bit, mark as scrolled to hide instruction
      if (scrollPosition > 100 && !hasScrolled) {
        setHasScrolled(true);
      }
      
      const categoryElements = categories.map(category => {
        const element = document.getElementById(category.slug);
        return {
          slug: category.slug,
          offsetTop: element?.offsetTop || 0
        };
      });
      
      // Add buffer to account for sticky header height
      const buffer = 150;
      
      for (let i = categoryElements.length - 1; i >= 0; i--) {
        if (scrollPosition >= categoryElements[i].offsetTop - buffer) {
          setActiveCategory(categoryElements[i].slug);
          break;
        }
      }
    };
    
    window.addEventListener("scroll", handleScroll);
    return () => window.removeEventListener("scroll", handleScroll);
  }, [categories, activeCategory, hasScrolled]);
  
  // Initialize scroll checking
  useEffect(() => {
    if (scrollContainerRef.current) {
      setTimeout(() => {
        checkScroll();
      }, 100);
    }
    
    // Check on resize too
    window.addEventListener('resize', checkScroll);
    return () => window.removeEventListener('resize', checkScroll);
  }, [categories]);
  
  const handleCategoryClick = (slug: string) => {
    // Toggle category selection - if same category clicked, deselect it
    const newSelectedCategory = selectedCategory === slug ? null : slug;
    
    setActiveCategory(slug);
    setHasScrolled(true);
    
    // Call parent callback for filtering
    if (onCategorySelect) {
      onCategorySelect(newSelectedCategory);
    }
    
    // Smooth scroll to the category if not filtering
    if (!onCategorySelect) {
      const element = document.getElementById(slug);
      if (element) {
        window.scrollTo({
          top: element.offsetTop - 120,
          behavior: 'smooth'
        });
      }
    }
  };
  
  // Modern chip-style navigation with Jouska design
  return (
    <div className="bg-gradient-to-r from-primary/95 to-primary dark:from-primary/90 dark:to-primary/95 shadow-md py-2">
      <div className="container mx-auto px-4">
        {/* Navigation wrapper with special styling to indicate scrolling */}
        <div className="relative">
          {/* Left shadow to indicate more on the left */}
          <div className={`absolute left-0 top-0 bottom-0 w-6 bg-gradient-to-r from-primary to-transparent dark:from-primary to-transparent z-10 pointer-events-none transition-opacity duration-300 ${canScrollLeft ? 'opacity-100' : 'opacity-0'}`}></div>
          
          {/* Right shadow to indicate more on the right */}
          <div className={`absolute right-0 top-0 bottom-0 w-6 bg-gradient-to-l from-primary to-transparent dark:from-primary to-transparent z-10 pointer-events-none transition-opacity duration-300 ${canScrollRight ? 'opacity-100' : 'opacity-0'}`}></div>
          
          {/* The actual scrollable container */}
          <div 
            ref={scrollContainerRef}
            onScroll={checkScroll}
            className="overflow-x-auto scrollbar-hide pb-1 -mx-2 px-2 scroll-smooth"
          >
            <div className="flex gap-2 items-center w-max pb-1 px-2">
              {/* "Tümü" button for showing all categories */}
              <button
                onClick={() => onCategorySelect && onCategorySelect(null)}
                className={`
                  whitespace-nowrap px-4 py-1.5 rounded-full text-sm font-medium transition-all duration-200
                  ${!selectedCategory 
                    ? 'bg-accent text-white shadow-lg transform scale-105 ring-2 ring-accent/30' 
                    : 'bg-white/10 text-white hover:bg-white/20 dark:bg-white/10 dark:hover:bg-white/20'
                  }
                `}
              >
                Tümü
              </button>
              
              {categories.map((category) => (
                <button
                  key={category.id}
                  onClick={() => handleCategoryClick(category.slug)}
                  className={`
                    whitespace-nowrap px-4 py-1.5 rounded-full text-sm font-medium transition-all duration-200
                    ${selectedCategory === category.slug 
                      ? 'bg-accent text-white shadow-lg transform scale-105 ring-2 ring-accent/30' 
                      : activeCategory === category.slug 
                      ? 'bg-background text-primary dark:text-primary shadow-md' 
                      : 'bg-white/10 text-white hover:bg-white/20 dark:bg-white/10 dark:hover:bg-white/20'
                    }
                  `}
                >
                  {category.name}
                </button>
              ))}
            </div>
          </div>
        </div>
        
        {/* Scroll indicator */}
        <div className="flex justify-center mt-0.5">
          <div className="w-12 h-1 bg-white/20 rounded-full"></div>
        </div>
        
        {/* Scroll instruction - shows only on initial load */}
        <div className={`text-center text-xs text-white/60 mt-0.5 transition-opacity duration-500 ${hasScrolled ? 'opacity-0 h-0 overflow-hidden' : 'opacity-100'}`}>
          <span className="inline-block animate-pulse">← Kategoriler için kaydır →</span>
        </div>

        {/* Manual navigation buttons for mobile */}
        <div className="flex justify-center mt-1 md:hidden">
          <button 
            onClick={() => {
              if (scrollContainerRef.current) {
                scrollContainerRef.current.scrollLeft -= 200;
                setTimeout(checkScroll, 100);
              }
            }}
            className={`bg-white/10 hover:bg-white/20 text-white p-1 rounded-full mr-2 transition-opacity duration-200 ${canScrollLeft ? 'opacity-100' : 'opacity-40'}`}
            disabled={!canScrollLeft}
          >
            <svg xmlns="http://www.w3.org/2000/svg" className="h-4 w-4" fill="none" viewBox="0 0 24 24" stroke="currentColor">
              <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M15 19l-7-7 7-7" />
            </svg>
          </button>
          
          <button 
            onClick={() => {
              if (scrollContainerRef.current) {
                scrollContainerRef.current.scrollLeft += 200;
                setTimeout(checkScroll, 100);
              }
            }}
            className={`bg-white/10 hover:bg-white/20 text-white p-1 rounded-full transition-opacity duration-200 ${canScrollRight ? 'opacity-100' : 'opacity-40'}`}
            disabled={!canScrollRight}
          >
            <svg xmlns="http://www.w3.org/2000/svg" className="h-4 w-4" fill="none" viewBox="0 0 24 24" stroke="currentColor">
              <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M9 5l7 7-7 7" />
            </svg>
          </button>
        </div>
      </div>
    </div>
  );
};
