import { useState } from "react";
import { Button } from "@/components/ui/button";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Badge } from "@/components/ui/badge";
import { Separator } from "@/components/ui/separator";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Textarea } from "@/components/ui/textarea";
import { useCart } from "@/hooks/use-cart";
import { CheckoutForm } from "@/components/payment/checkout-form";
import { ShoppingCart as CartIcon, Plus, Minus, Trash2, CreditCard } from "lucide-react";
import { motion, AnimatePresence } from "framer-motion";

interface ShoppingCartProps {
  isOpen: boolean;
  onClose: () => void;
}

export function ShoppingCart({ isOpen, onClose }: ShoppingCartProps) {
  const { items, updateQuantity, removeItem, updateNotes, clearCart, totalItems, totalPrice } = useCart();
  const [showCheckout, setShowCheckout] = useState(false);

  const handleCheckout = () => {
    if (items.length === 0) return;
    setShowCheckout(true);
  };

  const handleCheckoutClose = () => {
    setShowCheckout(false);
    alert("Siparişiniz başarıyla alındı! Teşekkür ederiz.");
  };

  if (!isOpen) return null;

  return (
    <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center p-4 z-50">
      <motion.div
        initial={{ opacity: 0, scale: 0.9 }}
        animate={{ opacity: 1, scale: 1 }}
        exit={{ opacity: 0, scale: 0.9 }}
        className="bg-white rounded-lg w-full max-w-2xl max-h-[90vh] overflow-hidden"
      >
        <div className="p-6 border-b">
          <div className="flex items-center justify-between">
            <div className="flex items-center space-x-2">
              <CartIcon className="w-6 h-6 text-orange-500" />
              <h2 className="text-xl font-bold">Sepetim</h2>
              {totalItems > 0 && (
                <Badge variant="secondary">{totalItems} ürün</Badge>
              )}
            </div>
            <Button variant="ghost" onClick={onClose}>✕</Button>
          </div>
        </div>

        <div className="flex-1 overflow-y-auto max-h-[60vh] p-6">
          {items.length === 0 ? (
            <div className="text-center py-12">
              <CartIcon className="w-16 h-16 text-gray-300 mx-auto mb-4" />
              <p className="text-gray-500">Sepetiniz boş</p>
              <p className="text-sm text-gray-400 mt-2">Menüden ürün ekleyerek başlayın</p>
            </div>
          ) : (
            <div className="space-y-4">
              <AnimatePresence>
                {items.map((item) => (
                  <motion.div
                    key={item.menuItem.id}
                    layout
                    initial={{ opacity: 0, x: -20 }}
                    animate={{ opacity: 1, x: 0 }}
                    exit={{ opacity: 0, x: 20 }}
                  >
                    <Card>
                      <CardContent className="p-4">
                        <div className="flex items-start justify-between">
                          <div className="flex-1">
                            <h3 className="font-medium">{item.menuItem.name}</h3>
                            <p className="text-sm text-gray-600 mt-1">
                              ₺{item.menuItem.price} × {item.quantity} = ₺{(parseFloat(item.menuItem.price) * item.quantity).toFixed(2)}
                            </p>
                            
                            {/* Quantity controls */}
                            <div className="flex items-center space-x-2 mt-3">
                              <Button
                                variant="outline"
                                size="sm"
                                onClick={() => updateQuantity(item.menuItem.id, item.quantity - 1)}
                              >
                                <Minus className="w-3 h-3" />
                              </Button>
                              <span className="w-8 text-center">{item.quantity}</span>
                              <Button
                                variant="outline"
                                size="sm"
                                onClick={() => updateQuantity(item.menuItem.id, item.quantity + 1)}
                              >
                                <Plus className="w-3 h-3" />
                              </Button>
                              <Button
                                variant="destructive"
                                size="sm"
                                onClick={() => removeItem(item.menuItem.id)}
                                className="ml-4"
                              >
                                <Trash2 className="w-3 h-3" />
                              </Button>
                            </div>

                            {/* Notes */}
                            <div className="mt-3">
                              <Input
                                placeholder="Özel not (opsiyonel)"
                                value={item.notes || ""}
                                onChange={(e) => updateNotes(item.menuItem.id, e.target.value)}
                                className="text-sm"
                              />
                            </div>
                          </div>
                        </div>
                      </CardContent>
                    </Card>
                  </motion.div>
                ))}
              </AnimatePresence>

            </div>
          )}
        </div>

        {items.length > 0 && (
          <div className="border-t p-6 bg-gray-50">
            <div className="flex items-center justify-between mb-4">
              <div className="text-lg font-bold">
                Toplam: ₺{totalPrice.toFixed(2)}
              </div>
              <Button
                variant="outline"
                onClick={clearCart}
                className="text-red-600"
              >
                Sepeti Temizle
              </Button>
            </div>
            
            <Button
              onClick={handleCheckout}
              className="w-full bg-orange-500 hover:bg-orange-600"
            >
              <CreditCard className="w-4 h-4 mr-2" />
              Sipariş Ver
            </Button>
          </div>
        )}
      </motion.div>
      
      {/* Checkout Form */}
      <CheckoutForm
        isOpen={showCheckout}
        onClose={handleCheckoutClose}
      />
    </div>
  );
}