import React, { useState } from "react";
import { Card, CardContent, CardDescription, CardFooter, CardHeader, CardTitle } from "@/components/ui/card";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Button } from "@/components/ui/button";
import { Switch } from "@/components/ui/switch";
import { useThemeSettings, useUpdateTheme, useCreateTheme, useAllThemes, useActivateTheme, useDeleteTheme } from "@/hooks/use-theme-settings";
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { Separator } from "@/components/ui/separator";
import { Loader2, RefreshCw, Trash2, Plus, Check } from "lucide-react";
import { cn } from "@/lib/utils";
import { ThemeSettings } from "@shared/schema";
import {
  Dialog,
  DialogContent,
  DialogDescription,
  DialogFooter,
  DialogHeader,
  DialogTitle,
  DialogTrigger,
} from "@/components/ui/dialog";

export default function ThemeCustomizer() {
  const [activeTab, setActiveTab] = useState("current");
  const [createDialogOpen, setCreateDialogOpen] = useState(false);
  const [selectedThemeId, setSelectedThemeId] = useState<number | null>(null);
  
  // Form state for new theme
  const [newThemeName, setNewThemeName] = useState("");
  const [newThemePrimary, setNewThemePrimary] = useState("#1E40AF");
  const [newThemeSecondary, setNewThemeSecondary] = useState("#3B82F6");
  const [newThemeAccent, setNewThemeAccent] = useState("#0EA5E9");
  const [newThemeText, setNewThemeText] = useState("#1E293B");
  const [newThemeBackground, setNewThemeBackground] = useState("#FFFFFF");
  const [newThemeBorderRadius, setNewThemeBorderRadius] = useState("0.5rem");
  const [newThemeDarkMode, setNewThemeDarkMode] = useState(false);
  
  // Form state for theme update
  const [editName, setEditName] = useState("");
  const [editPrimary, setEditPrimary] = useState("");
  const [editSecondary, setEditSecondary] = useState("");
  const [editAccent, setEditAccent] = useState("");
  const [editText, setEditText] = useState("");
  const [editBackground, setEditBackground] = useState("");
  const [editBorderRadius, setEditBorderRadius] = useState("");
  const [editDarkMode, setEditDarkMode] = useState(false);
  
  // Queries
  const { data: activeTheme, isLoading: isActiveThemeLoading } = useThemeSettings();
  const { data: allThemes, isLoading: isAllThemesLoading } = useAllThemes();
  
  // Mutations
  const updateTheme = useUpdateTheme();
  const createTheme = useCreateTheme();
  const activateTheme = useActivateTheme();
  const deleteTheme = useDeleteTheme();
  
  // Reset form when creation is successful
  React.useEffect(() => {
    if (createTheme.isSuccess) {
      setNewThemeName("");
      setNewThemePrimary("#3b82f6");
      setNewThemeSecondary("#64748b");
      setNewThemeAccent("#f59e0b");
      setNewThemeText("#1f2937");
      setNewThemeBackground("#ffffff");
      setNewThemeBorderRadius("0.5rem");
      setNewThemeDarkMode(false);
      setCreateDialogOpen(false);
    }
  }, [createTheme.isSuccess]);
  
  // Load theme data for editing
  const loadThemeForEdit = (theme: ThemeSettings) => {
    setSelectedThemeId(theme.id);
    setEditName(theme.name);
    setEditPrimary(theme.primaryColor || "#3b82f6");
    setEditSecondary(theme.secondaryColor || "#64748b");
    setEditAccent(theme.accentColor || "#f59e0b");
    setEditText(theme.textColor || "#1f2937");
    setEditBackground(theme.backgroundColor || "#ffffff");
    setEditBorderRadius("0.5rem");
    setEditDarkMode(false);
  };
  
  // Reset edit form
  const resetEditForm = () => {
    setSelectedThemeId(null);
    setEditName("");
    setEditPrimary("");
    setEditSecondary("");
    setEditAccent("");
    setEditText("");
    setEditBackground("");
    setEditBorderRadius("");
    setEditDarkMode(false);
  };
  
  // Update current theme
  const handleUpdateTheme = () => {
    if (!activeTheme || !selectedThemeId) return;
    
    updateTheme.mutate({
      id: selectedThemeId,
      data: {
        name: editName,
        primaryColor: editPrimary,
        secondaryColor: editSecondary,
        accentColor: editAccent,
        textColor: editText,
        backgroundColor: editBackground,
        borderRadius: editBorderRadius,
        darkMode: editDarkMode
      }
    });
  };
  
  // Create new theme
  const handleCreateTheme = () => {
    if (!newThemeName.trim()) return;
    
    createTheme.mutate({
      name: newThemeName,
      primaryColor: newThemePrimary,
      secondaryColor: newThemeSecondary,
      accentColor: newThemeAccent,
      textColor: newThemeText,
      backgroundColor: newThemeBackground,
      fontHeading: "Playfair Display",
      fontBody: "Plus Jakarta Sans",
      borderRadius: newThemeBorderRadius,
      isActive: false
    });
  };
  
  // Activate theme
  const handleActivateTheme = (themeId: number) => {
    activateTheme.mutate(themeId);
  };
  
  // Delete theme
  const handleDeleteTheme = (themeId: number) => {
    deleteTheme.mutate(themeId);
  };
  
  if (isActiveThemeLoading) {
    return (
      <div className="flex items-center justify-center p-8">
        <Loader2 className="h-8 w-8 animate-spin text-primary" />
      </div>
    );
  }
  
  // If no active theme is found, show a message
  if (!activeTheme) {
    return (
      <Card>
        <CardHeader>
          <CardTitle>Tema Özelleştirme</CardTitle>
          <CardDescription>Herhangi bir aktif tema bulunamadı.</CardDescription>
        </CardHeader>
        <CardFooter>
          <Dialog open={createDialogOpen} onOpenChange={setCreateDialogOpen}>
            <DialogTrigger asChild>
              <Button>Tema Oluştur</Button>
            </DialogTrigger>
            <DialogContent>
              <DialogHeader>
                <DialogTitle>Yeni Tema Oluştur</DialogTitle>
                <DialogDescription>Yeni bir tema oluşturmak için gerekli bilgileri doldurun.</DialogDescription>
              </DialogHeader>
              <div className="grid gap-4 py-4">
                <div className="grid grid-cols-4 items-center gap-4">
                  <Label htmlFor="name" className="text-right">Tema Adı</Label>
                  <Input
                    id="name"
                    value={newThemeName}
                    onChange={(e) => setNewThemeName(e.target.value)}
                    className="col-span-3"
                  />
                </div>
                <div className="grid grid-cols-4 items-center gap-4">
                  <Label htmlFor="primary" className="text-right">Ana Renk</Label>
                  <div className="col-span-3 flex gap-2">
                    <Input
                      id="primary"
                      type="color"
                      value={newThemePrimary}
                      onChange={(e) => setNewThemePrimary(e.target.value)}
                      className="w-12"
                    />
                    <Input
                      value={newThemePrimary}
                      onChange={(e) => setNewThemePrimary(e.target.value)}
                      className="flex-1"
                    />
                  </div>
                </div>
                <div className="grid grid-cols-4 items-center gap-4">
                  <Label htmlFor="secondary" className="text-right">İkincil Renk</Label>
                  <div className="col-span-3 flex gap-2">
                    <Input
                      id="secondary"
                      type="color"
                      value={newThemeSecondary}
                      onChange={(e) => setNewThemeSecondary(e.target.value)}
                      className="w-12"
                    />
                    <Input
                      value={newThemeSecondary}
                      onChange={(e) => setNewThemeSecondary(e.target.value)}
                      className="flex-1"
                    />
                  </div>
                </div>
                <div className="grid grid-cols-4 items-center gap-4">
                  <Label htmlFor="accent" className="text-right">Vurgu Rengi</Label>
                  <div className="col-span-3 flex gap-2">
                    <Input
                      id="accent"
                      type="color"
                      value={newThemeAccent}
                      onChange={(e) => setNewThemeAccent(e.target.value)}
                      className="w-12"
                    />
                    <Input
                      value={newThemeAccent}
                      onChange={(e) => setNewThemeAccent(e.target.value)}
                      className="flex-1"
                    />
                  </div>
                </div>
                <div className="grid grid-cols-4 items-center gap-4">
                  <Label htmlFor="darkMode" className="text-right">Karanlık Mod</Label>
                  <Switch
                    id="darkMode"
                    checked={newThemeDarkMode}
                    onCheckedChange={setNewThemeDarkMode}
                  />
                </div>
              </div>
              <DialogFooter>
                <Button variant="outline" onClick={() => setCreateDialogOpen(false)}>İptal</Button>
                <Button onClick={handleCreateTheme} disabled={!newThemeName}>Oluştur</Button>
              </DialogFooter>
            </DialogContent>
          </Dialog>
        </CardFooter>
      </Card>
    );
  }
  
  return (
    <Card className="w-full">
      <CardHeader>
        <CardTitle>Tema Özelleştirme</CardTitle>
        <CardDescription>Menü görünümü için tema ayarlarınızı özelleştirin.</CardDescription>
      </CardHeader>
      
      <Tabs defaultValue="current" value={activeTab} onValueChange={setActiveTab}>
        <div className="px-6">
          <TabsList className="grid w-full grid-cols-2">
            <TabsTrigger value="current">Aktif Tema</TabsTrigger>
            <TabsTrigger value="themes">Tüm Temalar</TabsTrigger>
          </TabsList>
        </div>
        
        <TabsContent value="current" className="p-6 pt-4">
          <div className="space-y-4">
            {selectedThemeId ? (
              <div className="grid gap-4">
                <div className="grid grid-cols-4 items-center gap-4">
                  <Label htmlFor="edit-name" className="text-right">Tema Adı</Label>
                  <Input
                    id="edit-name"
                    value={editName}
                    onChange={(e) => setEditName(e.target.value)}
                    className="col-span-3"
                  />
                </div>
                <div className="grid grid-cols-4 items-center gap-4">
                  <Label htmlFor="edit-primary" className="text-right">Ana Renk</Label>
                  <div className="col-span-3 flex gap-2">
                    <Input
                      id="edit-primary"
                      type="color"
                      value={editPrimary}
                      onChange={(e) => setEditPrimary(e.target.value)}
                      className="w-12"
                    />
                    <Input
                      value={editPrimary}
                      onChange={(e) => setEditPrimary(e.target.value)}
                      className="flex-1"
                    />
                  </div>
                </div>
                <div className="grid grid-cols-4 items-center gap-4">
                  <Label htmlFor="edit-secondary" className="text-right">İkincil Renk</Label>
                  <div className="col-span-3 flex gap-2">
                    <Input
                      id="edit-secondary"
                      type="color"
                      value={editSecondary}
                      onChange={(e) => setEditSecondary(e.target.value)}
                      className="w-12"
                    />
                    <Input
                      value={editSecondary}
                      onChange={(e) => setEditSecondary(e.target.value)}
                      className="flex-1"
                    />
                  </div>
                </div>
                <div className="grid grid-cols-4 items-center gap-4">
                  <Label htmlFor="edit-accent" className="text-right">Vurgu Rengi</Label>
                  <div className="col-span-3 flex gap-2">
                    <Input
                      id="edit-accent"
                      type="color"
                      value={editAccent}
                      onChange={(e) => setEditAccent(e.target.value)}
                      className="w-12"
                    />
                    <Input
                      value={editAccent}
                      onChange={(e) => setEditAccent(e.target.value)}
                      className="flex-1"
                    />
                  </div>
                </div>
                <div className="grid grid-cols-4 items-center gap-4">
                  <Label htmlFor="edit-text" className="text-right">Metin Rengi</Label>
                  <div className="col-span-3 flex gap-2">
                    <Input
                      id="edit-text"
                      type="color"
                      value={editText}
                      onChange={(e) => setEditText(e.target.value)}
                      className="w-12"
                    />
                    <Input
                      value={editText}
                      onChange={(e) => setEditText(e.target.value)}
                      className="flex-1"
                    />
                  </div>
                </div>
                <div className="grid grid-cols-4 items-center gap-4">
                  <Label htmlFor="edit-background" className="text-right">Arkaplan Rengi</Label>
                  <div className="col-span-3 flex gap-2">
                    <Input
                      id="edit-background"
                      type="color"
                      value={editBackground}
                      onChange={(e) => setEditBackground(e.target.value)}
                      className="w-12"
                    />
                    <Input
                      value={editBackground}
                      onChange={(e) => setEditBackground(e.target.value)}
                      className="flex-1"
                    />
                  </div>
                </div>
                <div className="grid grid-cols-4 items-center gap-4">
                  <Label htmlFor="edit-border-radius" className="text-right">Köşe Yuvarlaklığı</Label>
                  <Input
                    id="edit-border-radius"
                    value={editBorderRadius}
                    onChange={(e) => setEditBorderRadius(e.target.value)}
                    className="col-span-3"
                    placeholder="0.5rem"
                  />
                </div>
                <div className="grid grid-cols-4 items-center gap-4">
                  <Label htmlFor="edit-dark-mode" className="text-right">Karanlık Mod</Label>
                  <Switch
                    id="edit-dark-mode"
                    checked={editDarkMode}
                    onCheckedChange={setEditDarkMode}
                  />
                </div>
                
                <div className="flex justify-end gap-2 pt-4">
                  <Button variant="outline" onClick={resetEditForm}>İptal</Button>
                  <Button onClick={handleUpdateTheme} disabled={updateTheme.isPending}>
                    {updateTheme.isPending ? (
                      <>
                        <Loader2 className="mr-2 h-4 w-4 animate-spin" />
                        Kaydediliyor
                      </>
                    ) : "Değişiklikleri Kaydet"}
                  </Button>
                </div>
              </div>
            ) : (
              <>
                <div className="grid grid-cols-2 gap-4">
                  <div className="space-y-2">
                    <Label>Tema Adı</Label>
                    <div className="rounded-md border p-2">{activeTheme.name}</div>
                  </div>
                  <div className="space-y-2">
                    <Label>Font Ailesi</Label>
                    <div className="rounded-md border p-2">{activeTheme.fontFamily || "Ubuntu"}</div>
                  </div>
                  <div className="space-y-2">
                    <Label>Ana Renk</Label>
                    <div className="flex items-center gap-2">
                      <div className="h-6 w-6 rounded-md" style={{ backgroundColor: activeTheme.primaryColor || "#3b82f6" }} />
                      <div className="rounded-md border p-2 flex-1">{activeTheme.primaryColor || "#3b82f6"}</div>
                    </div>
                  </div>
                  <div className="space-y-2">
                    <Label>İkincil Renk</Label>
                    <div className="flex items-center gap-2">
                      <div className="h-6 w-6 rounded-md" style={{ backgroundColor: (activeTheme.secondaryColor || "#64748b") as string }} />
                      <div className="rounded-md border p-2 flex-1">{activeTheme.secondaryColor || "#64748b"}</div>
                    </div>
                  </div>
                  <div className="space-y-2">
                    <Label>Vurgu Rengi</Label>
                    <div className="flex items-center gap-2">
                      <div className="h-6 w-6 rounded-md" style={{ backgroundColor: (activeTheme.accentColor || "#f59e0b") as string }} />
                      <div className="rounded-md border p-2 flex-1">{activeTheme.accentColor || "#f59e0b"}</div>
                    </div>
                  </div>
                  <div className="space-y-2">
                    <Label>Metin Rengi</Label>
                    <div className="flex items-center gap-2">
                      <div className="h-6 w-6 rounded-md" style={{ backgroundColor: (activeTheme.textColor || "#1f2937") as string }} />
                      <div className="rounded-md border p-2 flex-1">{activeTheme.textColor || "#1f2937"}</div>
                    </div>
                  </div>
                  <div className="space-y-2">
                    <Label>Arkaplan Rengi</Label>
                    <div className="flex items-center gap-2">
                      <div className="h-6 w-6 rounded-md" style={{ backgroundColor: (activeTheme.backgroundColor || "#ffffff") as string }} />
                      <div className="rounded-md border p-2 flex-1">{activeTheme.backgroundColor || "#ffffff"}</div>
                    </div>
                  </div>
                </div>
                <div className="pt-4">
                  <Button 
                    onClick={() => loadThemeForEdit(activeTheme)}
                    className="w-full"
                  >
                    Temayı Düzenle
                  </Button>
                </div>
              </>
            )}
          </div>
        </TabsContent>
        
        <TabsContent value="themes" className="p-6 pt-2">
          {isAllThemesLoading ? (
            <div className="flex items-center justify-center p-8">
              <Loader2 className="h-8 w-8 animate-spin text-primary" />
            </div>
          ) : (
            <div className="space-y-4">
              <div className="flex justify-end">
                <Dialog open={createDialogOpen} onOpenChange={setCreateDialogOpen}>
                  <DialogTrigger asChild>
                    <Button>
                      <Plus className="h-4 w-4 mr-2" />
                      Yeni Tema Oluştur
                    </Button>
                  </DialogTrigger>
                  <DialogContent>
                    <DialogHeader>
                      <DialogTitle>Yeni Tema Oluştur</DialogTitle>
                      <DialogDescription>Yeni bir tema oluşturmak için gerekli bilgileri doldurun.</DialogDescription>
                    </DialogHeader>
                    <div className="grid gap-4 py-4">
                      <div className="grid grid-cols-4 items-center gap-4">
                        <Label htmlFor="name" className="text-right">Tema Adı</Label>
                        <Input
                          id="name"
                          value={newThemeName}
                          onChange={(e) => setNewThemeName(e.target.value)}
                          className="col-span-3"
                        />
                      </div>
                      <div className="grid grid-cols-4 items-center gap-4">
                        <Label htmlFor="primary" className="text-right">Ana Renk</Label>
                        <div className="col-span-3 flex gap-2">
                          <Input
                            id="primary"
                            type="color"
                            value={newThemePrimary}
                            onChange={(e) => setNewThemePrimary(e.target.value)}
                            className="w-12"
                          />
                          <Input
                            value={newThemePrimary}
                            onChange={(e) => setNewThemePrimary(e.target.value)}
                            className="flex-1"
                          />
                        </div>
                      </div>
                      <div className="grid grid-cols-4 items-center gap-4">
                        <Label htmlFor="secondary" className="text-right">İkincil Renk</Label>
                        <div className="col-span-3 flex gap-2">
                          <Input
                            id="secondary"
                            type="color"
                            value={newThemeSecondary}
                            onChange={(e) => setNewThemeSecondary(e.target.value)}
                            className="w-12"
                          />
                          <Input
                            value={newThemeSecondary}
                            onChange={(e) => setNewThemeSecondary(e.target.value)}
                            className="flex-1"
                          />
                        </div>
                      </div>
                      <div className="grid grid-cols-4 items-center gap-4">
                        <Label htmlFor="accent" className="text-right">Vurgu Rengi</Label>
                        <div className="col-span-3 flex gap-2">
                          <Input
                            id="accent"
                            type="color"
                            value={newThemeAccent}
                            onChange={(e) => setNewThemeAccent(e.target.value)}
                            className="w-12"
                          />
                          <Input
                            value={newThemeAccent}
                            onChange={(e) => setNewThemeAccent(e.target.value)}
                            className="flex-1"
                          />
                        </div>
                      </div>
                      <div className="grid grid-cols-4 items-center gap-4">
                        <Label htmlFor="darkMode" className="text-right">Karanlık Mod</Label>
                        <Switch
                          id="darkMode"
                          checked={newThemeDarkMode}
                          onCheckedChange={setNewThemeDarkMode}
                        />
                      </div>
                    </div>
                    <DialogFooter>
                      <Button variant="outline" onClick={() => setCreateDialogOpen(false)}>İptal</Button>
                      <Button onClick={handleCreateTheme} disabled={!newThemeName || createTheme.isPending}>
                        {createTheme.isPending ? (
                          <>
                            <Loader2 className="mr-2 h-4 w-4 animate-spin" />
                            Oluşturuluyor
                          </>
                        ) : "Oluştur"}
                      </Button>
                    </DialogFooter>
                  </DialogContent>
                </Dialog>
              </div>
              
              <div className="space-y-3">
                {allThemes?.map((theme) => (
                  <div key={theme.id} className={cn(
                    "flex items-center justify-between rounded-lg border p-3",
                    theme.isActive && "border-primary bg-primary/5"
                  )}>
                    <div className="flex items-center gap-3">
                      <div className="flex h-10 w-10 items-center justify-center rounded bg-primary/10">
                        <div 
                          className="h-6 w-6 rounded" 
                          style={{ background: `linear-gradient(135deg, ${theme.primaryColor}, ${theme.accentColor})` }}
                        />
                      </div>
                      <div>
                        <div className="font-medium">{theme.name}</div>
                        <div className="text-sm text-muted-foreground">Font: {theme.fontFamily || "Ubuntu"}</div>
                      </div>
                      {theme.isActive && (
                        <div className="ml-2 rounded-full bg-primary px-2 py-0.5 text-xs text-primary-foreground">
                          Aktif
                        </div>
                      )}
                    </div>
                    
                    <div className="flex items-center gap-2">
                      {!theme.isActive && (
                        <Button
                          variant="outline"
                          size="sm"
                          onClick={() => handleActivateTheme(theme.id)}
                          disabled={activateTheme.isPending}
                        >
                          {activateTheme.isPending && activateTheme.variables === theme.id ? (
                            <Loader2 className="h-4 w-4 animate-spin" />
                          ) : (
                            <Check className="h-4 w-4" />
                          )}
                          <span className="ml-1.5">Aktif Et</span>
                        </Button>
                      )}
                      
                      <Button
                        variant="ghost"
                        size="sm"
                        onClick={() => {
                          loadThemeForEdit(theme);
                          setActiveTab("current");
                        }}
                      >
                        Düzenle
                      </Button>
                      
                      {!theme.isActive && (
                        <Button
                          variant="ghost"
                          size="sm"
                          className="text-destructive hover:text-destructive"
                          onClick={() => handleDeleteTheme(theme.id)}
                          disabled={deleteTheme.isPending}
                        >
                          {deleteTheme.isPending && deleteTheme.variables === theme.id ? (
                            <Loader2 className="h-4 w-4 animate-spin" />
                          ) : (
                            <Trash2 className="h-4 w-4" />
                          )}
                        </Button>
                      )}
                    </div>
                  </div>
                ))}
                
                {allThemes?.length === 0 && (
                  <div className="text-center py-8 text-muted-foreground">
                    <div>Henüz tema oluşturulmadı.</div>
                    <Button
                      variant="outline"
                      size="sm"
                      className="mt-2"
                      onClick={() => setCreateDialogOpen(true)}
                    >
                      <Plus className="h-4 w-4 mr-2" />
                      Yeni Tema Oluştur
                    </Button>
                  </div>
                )}
              </div>
            </div>
          )}
        </TabsContent>
      </Tabs>
    </Card>
  );
}