import React, { useState } from "react";
import { Button } from "@/components/ui/button";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs";
import { MenuItem, Category } from "@shared/schema";
import { 
  BarChart, 
  Bar, 
  XAxis, 
  YAxis, 
  CartesianGrid, 
  Tooltip, 
  Legend, 
  ResponsiveContainer,
  PieChart,
  Pie,
  Cell
} from "recharts";
import { ArrowUp, ArrowDown, TrendingUp, Bookmark, Clock, Heart, Search, Filter } from "lucide-react";

interface StatsAnalyticsProps {
  menuItems: MenuItem[];
  categories: Category[];
}

export const StatsAnalytics = ({ menuItems, categories }: StatsAnalyticsProps) => {
  const [timeRange, setTimeRange] = useState("monthly");
  
  // Kategori bazlı menü öğeleri sayısı
  const categoryStats = categories.map(category => {
    const itemCount = menuItems.filter(item => item.categoryId === category.id).length;
    return {
      name: category.name,
      count: itemCount,
      fill: getRandomColor(category.id)
    };
  }).sort((a, b) => b.count - a.count);
  
  // Özel işaretlere sahip menü öğeleri sayısı
  const specialTagsStats = [
    {
      name: "Vejetaryen",
      count: menuItems.filter(item => item.isVegetarian).length,
      fill: "#4ade80"
    },
    {
      name: "Şef Özel",
      count: menuItems.filter(item => item.isChefSpecial).length,
      fill: "#f59e0b"
    },
    {
      name: "Baharatlı",
      count: menuItems.filter(item => item.isSpicy).length,
      fill: "#ef4444"
    },
    {
      name: "Aylık Özel",
      count: menuItems.filter(item => item.isMonthlySpecial).length,
      fill: "#8b5cf6"
    },
    {
      name: "Kampanya",
      count: menuItems.filter(item => item.isCampaign).length,
      fill: "#ec4899"
    }
  ].filter(stat => stat.count > 0);
  
  // Popüler menü öğeleri (şu an için şef özel ve aylık özel öğelerini gösterir)
  const popularItems = menuItems
    .filter(item => item.isChefSpecial || item.isMonthlySpecial)
    .map(item => ({
      name: item.name,
      views: Math.floor(Math.random() * 1000) + 50, // Simülasyon veri
      favorites: Math.floor(Math.random() * 100), // Simülasyon veri
      fill: getRandomColor(item.id)
    }))
    .sort((a, b) => b.views - a.views)
    .slice(0, 5);
  
  // Menü fiyat aralıkları
  const priceRanges = getPriceRangeStats();
  
  // Fiyat aralığı istatistiklerini hesapla
  function getPriceRangeStats() {
    const ranges = [
      { name: "0-50 TL", min: 0, max: 50, count: 0, fill: "#0ea5e9" },
      { name: "51-100 TL", min: 51, max: 100, count: 0, fill: "#4ade80" },
      { name: "101-150 TL", min: 101, max: 150, count: 0, fill: "#f59e0b" },
      { name: "151-200 TL", min: 151, max: 200, count: 0, fill: "#ef4444" },
      { name: "200+ TL", min: 201, max: Infinity, count: 0, fill: "#8b5cf6" }
    ];
    
    menuItems.forEach(item => {
      // Fiyat format: "100 TL", "Varies", vb.
      const price = parseFloat(item.price.replace(/[^0-9.]/g, ""));
      if (!isNaN(price)) {
        const range = ranges.find(r => price >= r.min && price <= r.max);
        if (range) range.count++;
      }
    });
    
    return ranges.filter(range => range.count > 0);
  }
  
  // Renk üret
  function getRandomColor(seed: number) {
    const colors = [
      "#0ea5e9", "#4ade80", "#f59e0b", "#ef4444", "#8b5cf6", 
      "#ec4899", "#06b6d4", "#84cc16", "#14b8a6", "#3b82f6"
    ];
    return colors[seed % colors.length];
  }
  
  return (
    <div className="space-y-6">
      <div className="flex justify-between items-center">
        <h2 className="text-2xl font-bold text-primary font-display">İstatistikler ve Analizler</h2>
        <div className="flex space-x-2">
          <Select value={timeRange} onValueChange={setTimeRange}>
            <SelectTrigger className="w-[180px]">
              <SelectValue placeholder="Zaman Aralığı" />
            </SelectTrigger>
            <SelectContent>
              <SelectItem value="weekly">Haftalık</SelectItem>
              <SelectItem value="monthly">Aylık</SelectItem>
              <SelectItem value="yearly">Yıllık</SelectItem>
              <SelectItem value="all">Tüm Zamanlar</SelectItem>
            </SelectContent>
          </Select>
        </div>
      </div>
      
      {/* Özet kartlar */}
      <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-4">
        <Card>
          <CardContent className="pt-6">
            <div className="flex justify-between items-start">
              <div>
                <p className="text-sm font-medium text-muted-foreground">Toplam Menü Öğesi</p>
                <h3 className="text-2xl font-bold mt-1">{menuItems.length}</h3>
              </div>
              <div className="p-2 bg-primary/10 rounded-full text-primary">
                <Filter className="h-5 w-5" />
              </div>
            </div>
            <div className="flex items-center mt-4 text-xs">
              <div className="flex items-center text-green-500">
                <ArrowUp className="h-3 w-3 mr-1" />
                <span>12%</span>
              </div>
              <span className="text-muted-foreground ml-2">Son {timeRange === 'weekly' ? 'hafta' : timeRange === 'monthly' ? 'ay' : 'yıl'}</span>
            </div>
          </CardContent>
        </Card>
        
        <Card>
          <CardContent className="pt-6">
            <div className="flex justify-between items-start">
              <div>
                <p className="text-sm font-medium text-muted-foreground">Kategoriler</p>
                <h3 className="text-2xl font-bold mt-1">{categories.length}</h3>
              </div>
              <div className="p-2 bg-primary/10 rounded-full text-primary">
                <Search className="h-5 w-5" />
              </div>
            </div>
            <div className="flex items-center mt-4 text-xs">
              <div className="flex items-center text-green-500">
                <ArrowUp className="h-3 w-3 mr-1" />
                <span>5%</span>
              </div>
              <span className="text-muted-foreground ml-2">Son {timeRange === 'weekly' ? 'hafta' : timeRange === 'monthly' ? 'ay' : 'yıl'}</span>
            </div>
          </CardContent>
        </Card>
        
        <Card>
          <CardContent className="pt-6">
            <div className="flex justify-between items-start">
              <div>
                <p className="text-sm font-medium text-muted-foreground">Günlük Görüntülenme</p>
                <h3 className="text-2xl font-bold mt-1">1,248</h3>
              </div>
              <div className="p-2 bg-primary/10 rounded-full text-primary">
                <TrendingUp className="h-5 w-5" />
              </div>
            </div>
            <div className="flex items-center mt-4 text-xs">
              <div className="flex items-center text-green-500">
                <ArrowUp className="h-3 w-3 mr-1" />
                <span>18%</span>
              </div>
              <span className="text-muted-foreground ml-2">Son {timeRange === 'weekly' ? 'hafta' : timeRange === 'monthly' ? 'ay' : 'yıl'}</span>
            </div>
          </CardContent>
        </Card>
        
        <Card>
          <CardContent className="pt-6">
            <div className="flex justify-between items-start">
              <div>
                <p className="text-sm font-medium text-muted-foreground">Favorilere Eklenenler</p>
                <h3 className="text-2xl font-bold mt-1">267</h3>
              </div>
              <div className="p-2 bg-primary/10 rounded-full text-primary">
                <Heart className="h-5 w-5" />
              </div>
            </div>
            <div className="flex items-center mt-4 text-xs">
              <div className="flex items-center text-red-500">
                <ArrowDown className="h-3 w-3 mr-1" />
                <span>3%</span>
              </div>
              <span className="text-muted-foreground ml-2">Son {timeRange === 'weekly' ? 'hafta' : timeRange === 'monthly' ? 'ay' : 'yıl'}</span>
            </div>
          </CardContent>
        </Card>
      </div>
      
      <Tabs defaultValue="categories" className="w-full">
        <TabsList>
          <TabsTrigger value="categories">Kategori Dağılımı</TabsTrigger>
          <TabsTrigger value="popular">Popüler Ürünler</TabsTrigger>
          <TabsTrigger value="tags">Özel İşaretler</TabsTrigger>
          <TabsTrigger value="prices">Fiyat Aralıkları</TabsTrigger>
        </TabsList>
        
        <TabsContent value="categories" className="mt-6">
          <Card>
            <CardHeader>
              <CardTitle>Kategori Bazlı Menü Dağılımı</CardTitle>
              <CardDescription>
                Her kategoride kaç menü öğesi bulunduğunu gösterir.
              </CardDescription>
            </CardHeader>
            <CardContent>
              <div className="h-80">
                <ResponsiveContainer width="100%" height="100%">
                  <BarChart
                    data={categoryStats}
                    margin={{ top: 20, right: 30, left: 20, bottom: 5 }}
                  >
                    <CartesianGrid strokeDasharray="3 3" />
                    <XAxis dataKey="name" />
                    <YAxis />
                    <Tooltip />
                    <Legend />
                    <Bar dataKey="count" name="Menü Öğeleri" fill="#3b82f6" />
                  </BarChart>
                </ResponsiveContainer>
              </div>
            </CardContent>
          </Card>
        </TabsContent>
        
        <TabsContent value="popular" className="mt-6">
          <Card>
            <CardHeader>
              <CardTitle>En Popüler Menü Öğeleri</CardTitle>
              <CardDescription>
                En çok görüntülenen ve favorilere eklenen menü öğeleri.
              </CardDescription>
            </CardHeader>
            <CardContent>
              <div className="h-80">
                <ResponsiveContainer width="100%" height="100%">
                  <BarChart
                    data={popularItems}
                    margin={{ top: 20, right: 30, left: 20, bottom: 5 }}
                    layout="vertical"
                  >
                    <CartesianGrid strokeDasharray="3 3" />
                    <XAxis type="number" />
                    <YAxis dataKey="name" type="category" width={100} />
                    <Tooltip />
                    <Legend />
                    <Bar dataKey="views" name="Görüntülenme" fill="#3b82f6" />
                    <Bar dataKey="favorites" name="Favorilere Eklenme" fill="#f59e0b" />
                  </BarChart>
                </ResponsiveContainer>
              </div>
            </CardContent>
          </Card>
        </TabsContent>
        
        <TabsContent value="tags" className="mt-6">
          <Card>
            <CardHeader>
              <CardTitle>Özel İşaretlere Göre Dağılım</CardTitle>
              <CardDescription>
                Vejetaryen, baharatlı, şef özel vb. işaretlere sahip menü öğelerinin dağılımı.
              </CardDescription>
            </CardHeader>
            <CardContent>
              <div className="h-80 flex items-center justify-center">
                {specialTagsStats.length > 0 ? (
                  <ResponsiveContainer width="100%" height="100%">
                    <PieChart>
                      <Pie
                        data={specialTagsStats}
                        cx="50%"
                        cy="50%"
                        labelLine={true}
                        outerRadius={80}
                        fill="#8884d8"
                        dataKey="count"
                        nameKey="name"
                        label={(entry) => entry.name}
                      >
                        {specialTagsStats.map((entry, index) => (
                          <Cell key={`cell-${index}`} fill={entry.fill} />
                        ))}
                      </Pie>
                      <Tooltip />
                      <Legend />
                    </PieChart>
                  </ResponsiveContainer>
                ) : (
                  <div className="text-center text-gray-500">
                    <p>Henüz özel işaretlere sahip menü öğesi bulunmuyor.</p>
                  </div>
                )}
              </div>
            </CardContent>
          </Card>
        </TabsContent>
        
        <TabsContent value="prices" className="mt-6">
          <Card>
            <CardHeader>
              <CardTitle>Fiyat Aralıklarına Göre Dağılım</CardTitle>
              <CardDescription>
                Menü öğelerinin fiyat aralıklarına göre dağılımı.
              </CardDescription>
            </CardHeader>
            <CardContent>
              <div className="h-80">
                <ResponsiveContainer width="100%" height="100%">
                  <BarChart
                    data={priceRanges}
                    margin={{ top: 20, right: 30, left: 20, bottom: 5 }}
                  >
                    <CartesianGrid strokeDasharray="3 3" />
                    <XAxis dataKey="name" />
                    <YAxis />
                    <Tooltip />
                    <Legend />
                    <Bar dataKey="count" name="Menü Öğeleri" fill="#8884d8">
                      {priceRanges.map((entry, index) => (
                        <Cell key={`cell-${index}`} fill={entry.fill} />
                      ))}
                    </Bar>
                  </BarChart>
                </ResponsiveContainer>
              </div>
            </CardContent>
          </Card>
        </TabsContent>
      </Tabs>
    </div>
  );
};