import React, { useState } from "react";
import { useQuery, useMutation, useQueryClient } from "@tanstack/react-query";
import { useForm } from "react-hook-form";
import { zodResolver } from "@hookform/resolvers/zod";
import { z } from "zod";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Textarea } from "@/components/ui/textarea";
import { Switch } from "@/components/ui/switch";
import { Label } from "@/components/ui/label";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs";
import { Separator } from "@/components/ui/separator";
import { Badge } from "@/components/ui/badge";
import { useToast } from "@/hooks/use-toast";
import { SeoSettings as SeoSettingsType, insertSeoSettingsSchema } from "@shared/schema";
import { apiRequest } from "@/lib/queryClient";
import { 
  Search, 
  Globe, 
  Share2, 
  Twitter, 
  Facebook, 
  Code, 
  Eye,
  AlertCircle,
  CheckCircle,
  Settings
} from "lucide-react";

const seoFormSchema = insertSeoSettingsSchema.extend({
  pageTitle: z.string().min(1, "Sayfa başlığı gereklidir").max(60, "Başlık 60 karakterden kısa olmalıdır"),
  metaDescription: z.string().optional().refine(
    (val) => !val || val.length <= 160,
    "Meta açıklama 160 karakterden kısa olmalıdır"
  ),
  metaKeywords: z.string().optional(),
  ogTitle: z.string().optional().refine(
    (val) => !val || val.length <= 95,
    "OG başlık 95 karakterden kısa olmalıdır"
  ),
  ogDescription: z.string().optional().refine(
    (val) => !val || val.length <= 200,
    "OG açıklama 200 karakterden kısa olmalıdır"
  ),
  twitterTitle: z.string().optional().refine(
    (val) => !val || val.length <= 70,
    "Twitter başlık 70 karakterden kısa olmalıdır"
  ),
});

type SeoFormData = z.infer<typeof seoFormSchema>;

export function SeoSettings() {
  const { toast } = useToast();
  const queryClient = useQueryClient();
  const [previewMode, setPreviewMode] = useState<"google" | "facebook" | "twitter">("google");

  const { data: seoSettings, isLoading } = useQuery<SeoSettingsType | null>({
    queryKey: ["/api/seo"],
    queryFn: async () => {
      try {
        const response = await fetch("/api/seo");
        if (!response.ok) {
          if (response.status === 404) {
            return null;
          }
          throw new Error("SEO ayarları alınamadı");
        }
        return response.json();
      } catch (error) {
        console.error("SEO settings fetch error:", error);
        return null;
      }
    },
  });

  const form = useForm<SeoFormData>({
    resolver: zodResolver(seoFormSchema),
    defaultValues: {
      pageTitle: seoSettings?.pageTitle || "",
      metaDescription: seoSettings?.metaDescription || "",
      metaKeywords: seoSettings?.metaKeywords || "",
      ogTitle: seoSettings?.ogTitle || "",
      ogDescription: seoSettings?.ogDescription || "",
      ogImage: seoSettings?.ogImage || "",
      twitterTitle: seoSettings?.twitterTitle || "",
      twitterDescription: seoSettings?.twitterDescription || "",
      twitterImage: seoSettings?.twitterImage || "",
      canonicalUrl: seoSettings?.canonicalUrl || "",
      robotsContent: seoSettings?.robotsContent || "index, follow",
      structuredData: seoSettings?.structuredData || "",
      isActive: seoSettings?.isActive ?? true,
    },
  });

  React.useEffect(() => {
    if (seoSettings) {
      form.reset({
        pageTitle: seoSettings.pageTitle || "",
        metaDescription: seoSettings.metaDescription || "",
        metaKeywords: seoSettings.metaKeywords || "",
        ogTitle: seoSettings.ogTitle || "",
        ogDescription: seoSettings.ogDescription || "",
        ogImage: seoSettings.ogImage || "",
        twitterTitle: seoSettings.twitterTitle || "",
        twitterDescription: seoSettings.twitterDescription || "",
        twitterImage: seoSettings.twitterImage || "",
        canonicalUrl: seoSettings.canonicalUrl || "",
        robotsContent: seoSettings.robotsContent || "index, follow",
        structuredData: seoSettings.structuredData || "",
        isActive: seoSettings.isActive ?? true,
      });
    }
  }, [seoSettings, form]);

  const updateSeoMutation = useMutation({
    mutationFn: async (data: SeoFormData) => {
      if (seoSettings?.id) {
        const response = await apiRequest("PATCH", `/api/seo/${seoSettings.id}`, data);
        return response.json();
      } else {
        const response = await apiRequest("POST", "/api/seo", data);
        return response.json();
      }
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/seo"] });
      toast({
        title: "Başarılı",
        description: "SEO ayarları güncellendi",
      });
    },
    onError: (error: Error) => {
      toast({
        title: "Hata",
        description: error.message,
        variant: "destructive",
      });
    },
  });

  const onSubmit = (data: SeoFormData) => {
    updateSeoMutation.mutate(data);
  };

  const watchedValues = form.watch();

  const getCharacterCount = (text: string | undefined, maxLength: number) => {
    const length = text?.length || 0;
    const isOverLimit = length > maxLength;
    return (
      <span className={`text-xs ${isOverLimit ? 'text-red-500' : 'text-gray-500'}`}>
        {length}/{maxLength}
      </span>
    );
  };

  const getPreviewContent = () => {
    const title = watchedValues.pageTitle || "Jouska Cafe & Restaurant";
    const description = watchedValues.metaDescription || "Lezzetli yemekler ve taze kahvelerle sizleri bekliyoruz";
    const ogTitle = watchedValues.ogTitle || title;
    const ogDescription = watchedValues.ogDescription || description;
    const twitterTitle = watchedValues.twitterTitle || title;
    const twitterDescription = watchedValues.twitterDescription || description;

    switch (previewMode) {
      case "google":
        return (
          <div className="p-4 bg-white border rounded-lg">
            <div className="text-blue-600 text-lg hover:underline cursor-pointer">
              {title}
            </div>
            <div className="text-green-700 text-sm">
              {watchedValues.canonicalUrl || "https://jouskamenu.com"}
            </div>
            <div className="text-gray-700 text-sm mt-1">
              {description}
            </div>
          </div>
        );
      case "facebook":
        return (
          <div className="p-4 bg-white border rounded-lg max-w-sm">
            <div className="bg-gray-200 h-32 rounded mb-2 flex items-center justify-center">
              {watchedValues.ogImage ? (
                <img src={watchedValues.ogImage} alt="OG" className="h-full w-full object-cover rounded" />
              ) : (
                <span className="text-gray-500">Görsel</span>
              )}
            </div>
            <div className="text-sm font-medium">{ogTitle}</div>
            <div className="text-xs text-gray-600 mt-1">{ogDescription}</div>
            <div className="text-xs text-gray-500 mt-1">
              {watchedValues.canonicalUrl || "jouskamenu.com"}
            </div>
          </div>
        );
      case "twitter":
        return (
          <div className="p-4 bg-white border rounded-lg max-w-sm">
            <div className="bg-gray-200 h-32 rounded mb-2 flex items-center justify-center">
              {watchedValues.twitterImage ? (
                <img src={watchedValues.twitterImage} alt="Twitter" className="h-full w-full object-cover rounded" />
              ) : (
                <span className="text-gray-500">Görsel</span>
              )}
            </div>
            <div className="text-sm font-medium">{twitterTitle}</div>
            <div className="text-xs text-gray-600 mt-1">{twitterDescription}</div>
            <div className="text-xs text-gray-500 mt-1">
              {watchedValues.canonicalUrl || "jouskamenu.com"}
            </div>
          </div>
        );
      default:
        return null;
    }
  };

  if (isLoading) {
    return (
      <div className="flex items-center justify-center p-8">
        <div className="animate-spin rounded-full h-8 w-8 border-b-2 border-primary"></div>
      </div>
    );
  }

  return (
    <div className="space-y-6">
      <div className="flex items-center justify-between">
        <div>
          <h2 className="text-2xl font-bold">SEO Yönetimi</h2>
          <p className="text-gray-600 mt-1">
            Arama motoru optimizasyonu ve sosyal medya ayarlarını yönetin
          </p>
        </div>
        <Badge variant={seoSettings?.isActive ? "default" : "secondary"}>
          {seoSettings?.isActive ? "Aktif" : "Pasif"}
        </Badge>
      </div>

      <div className="grid grid-cols-1 lg:grid-cols-2 gap-6">
        {/* Settings Form */}
        <div>
          <Card>
            <CardHeader>
              <CardTitle className="flex items-center gap-2">
                <Settings className="h-5 w-5" />
                SEO Ayarları
              </CardTitle>
              <CardDescription>
                Web sitenizin arama motorlarında ve sosyal medyada nasıl görüneceğini ayarlayın
              </CardDescription>
            </CardHeader>
            <CardContent>
              <form onSubmit={form.handleSubmit(onSubmit)} className="space-y-6">
                <Tabs defaultValue="basic" className="w-full">
                  <TabsList className="grid w-full grid-cols-4">
                    <TabsTrigger value="basic">
                      <Search className="h-4 w-4 mr-1" />
                      Temel
                    </TabsTrigger>
                    <TabsTrigger value="social">
                      <Share2 className="h-4 w-4 mr-1" />
                      Sosyal
                    </TabsTrigger>
                    <TabsTrigger value="technical">
                      <Code className="h-4 w-4 mr-1" />
                      Teknik
                    </TabsTrigger>
                    <TabsTrigger value="preview">
                      <Eye className="h-4 w-4 mr-1" />
                      Önizleme
                    </TabsTrigger>
                  </TabsList>

                  <TabsContent value="basic" className="space-y-4">
                    <div>
                      <Label htmlFor="pageTitle">Sayfa Başlığı *</Label>
                      <div className="mt-1">
                        <Input
                          id="pageTitle"
                          {...form.register("pageTitle")}
                          placeholder="Jouska Cafe & Restaurant - Lezzetli Yemekler"
                        />
                        <div className="flex justify-between mt-1">
                          {form.formState.errors.pageTitle && (
                            <span className="text-red-500 text-xs">
                              {form.formState.errors.pageTitle.message}
                            </span>
                          )}
                          {getCharacterCount(watchedValues.pageTitle, 60)}
                        </div>
                      </div>
                    </div>

                    <div>
                      <Label htmlFor="metaDescription">Meta Açıklama</Label>
                      <div className="mt-1">
                        <Textarea
                          id="metaDescription"
                          {...form.register("metaDescription")}
                          placeholder="Taze kahveler, nefis yemekler ve sıcak atmosferin keyfini çıkarın..."
                          rows={3}
                        />
                        <div className="flex justify-between mt-1">
                          {form.formState.errors.metaDescription && (
                            <span className="text-red-500 text-xs">
                              {form.formState.errors.metaDescription.message}
                            </span>
                          )}
                          {getCharacterCount(watchedValues.metaDescription, 160)}
                        </div>
                      </div>
                    </div>

                    <div>
                      <Label htmlFor="metaKeywords">Anahtar Kelimeler</Label>
                      <Input
                        id="metaKeywords"
                        {...form.register("metaKeywords")}
                        placeholder="jouska, cafe, restaurant, kahve, yemek, kahvaltı"
                        className="mt-1"
                      />
                      <p className="text-xs text-gray-500 mt-1">
                        Virgülle ayırarak yazın (örn: cafe, restaurant, kahve)
                      </p>
                    </div>

                    <div>
                      <Label htmlFor="canonicalUrl">Canonical URL</Label>
                      <Input
                        id="canonicalUrl"
                        {...form.register("canonicalUrl")}
                        placeholder="https://jouskamenu.com"
                        className="mt-1"
                      />
                    </div>
                  </TabsContent>

                  <TabsContent value="social" className="space-y-4">
                    <div className="space-y-4">
                      <div className="flex items-center gap-2 text-sm font-medium">
                        <Facebook className="h-4 w-4 text-blue-600" />
                        Open Graph (Facebook, LinkedIn)
                      </div>
                      
                      <div>
                        <Label htmlFor="ogTitle">OG Başlık</Label>
                        <div className="mt-1">
                          <Input
                            id="ogTitle"
                            {...form.register("ogTitle")}
                            placeholder="Jouska Cafe & Restaurant - Eşsiz Lezzetler"
                          />
                          <div className="flex justify-between mt-1">
                            {form.formState.errors.ogTitle && (
                              <span className="text-red-500 text-xs">
                                {form.formState.errors.ogTitle.message}
                              </span>
                            )}
                            {getCharacterCount(watchedValues.ogTitle, 95)}
                          </div>
                        </div>
                      </div>

                      <div>
                        <Label htmlFor="ogDescription">OG Açıklama</Label>
                        <div className="mt-1">
                          <Textarea
                            id="ogDescription"
                            {...form.register("ogDescription")}
                            placeholder="Her lokmada kalite ve lezzeti bir arada yaşayın..."
                            rows={3}
                          />
                          <div className="flex justify-between mt-1">
                            {form.formState.errors.ogDescription && (
                              <span className="text-red-500 text-xs">
                                {form.formState.errors.ogDescription.message}
                              </span>
                            )}
                            {getCharacterCount(watchedValues.ogDescription, 200)}
                          </div>
                        </div>
                      </div>

                      <div>
                        <Label htmlFor="ogImage">OG Görsel URL</Label>
                        <Input
                          id="ogImage"
                          {...form.register("ogImage")}
                          placeholder="https://jouskamenu.com/images/og-image.jpg"
                          className="mt-1"
                        />
                      </div>
                    </div>

                    <Separator />

                    <div className="space-y-4">
                      <div className="flex items-center gap-2 text-sm font-medium">
                        <Twitter className="h-4 w-4 text-blue-400" />
                        Twitter Cards
                      </div>

                      <div>
                        <Label htmlFor="twitterTitle">Twitter Başlık</Label>
                        <div className="mt-1">
                          <Input
                            id="twitterTitle"
                            {...form.register("twitterTitle")}
                            placeholder="Jouska Cafe & Restaurant"
                          />
                          <div className="flex justify-between mt-1">
                            {form.formState.errors.twitterTitle && (
                              <span className="text-red-500 text-xs">
                                {form.formState.errors.twitterTitle.message}
                              </span>
                            )}
                            {getCharacterCount(watchedValues.twitterTitle, 70)}
                          </div>
                        </div>
                      </div>

                      <div>
                        <Label htmlFor="twitterDescription">Twitter Açıklama</Label>
                        <Textarea
                          id="twitterDescription"
                          {...form.register("twitterDescription")}
                          placeholder="Taze kahveler ve nefis yemeklerle Jouska deneyimi."
                          rows={2}
                          className="mt-1"
                        />
                      </div>

                      <div>
                        <Label htmlFor="twitterImage">Twitter Görsel URL</Label>
                        <Input
                          id="twitterImage"
                          {...form.register("twitterImage")}
                          placeholder="https://jouskamenu.com/images/twitter-image.jpg"
                          className="mt-1"
                        />
                      </div>
                    </div>
                  </TabsContent>

                  <TabsContent value="technical" className="space-y-4">
                    <div>
                      <Label htmlFor="robotsContent">Robots Meta İçeriği</Label>
                      <Input
                        id="robotsContent"
                        {...form.register("robotsContent")}
                        placeholder="index, follow"
                        className="mt-1"
                      />
                      <p className="text-xs text-gray-500 mt-1">
                        Örnekler: "index, follow", "noindex, nofollow", "index, nofollow"
                      </p>
                    </div>

                    <div>
                      <Label htmlFor="structuredData">Yapılandırılmış Veri (JSON-LD)</Label>
                      <Textarea
                        id="structuredData"
                        {...form.register("structuredData")}
                        placeholder='{"@context": "https://schema.org", "@type": "Restaurant", ...}'
                        rows={6}
                        className="mt-1 font-mono text-sm"
                      />
                      <p className="text-xs text-gray-500 mt-1">
                        Schema.org JSON-LD formatında yapılandırılmış veri
                      </p>
                    </div>

                    <div className="flex items-center space-x-2">
                      <Switch
                        id="isActive"
                        checked={form.watch("isActive") ?? true}
                        onCheckedChange={(checked) => form.setValue("isActive", checked)}
                      />
                      <Label htmlFor="isActive">SEO ayarlarını aktif et</Label>
                    </div>
                  </TabsContent>

                  <TabsContent value="preview" className="space-y-4">
                    <div>
                      <Label>Önizleme Modu</Label>
                      <div className="flex gap-2 mt-2">
                        <Button
                          type="button"
                          variant={previewMode === "google" ? "default" : "outline"}
                          size="sm"
                          onClick={() => setPreviewMode("google")}
                        >
                          <Search className="h-4 w-4 mr-1" />
                          Google
                        </Button>
                        <Button
                          type="button"
                          variant={previewMode === "facebook" ? "default" : "outline"}
                          size="sm"
                          onClick={() => setPreviewMode("facebook")}
                        >
                          <Facebook className="h-4 w-4 mr-1" />
                          Facebook
                        </Button>
                        <Button
                          type="button"
                          variant={previewMode === "twitter" ? "default" : "outline"}
                          size="sm"
                          onClick={() => setPreviewMode("twitter")}
                        >
                          <Twitter className="h-4 w-4 mr-1" />
                          Twitter
                        </Button>
                      </div>
                    </div>

                    <div>
                      <Label>Önizleme</Label>
                      <div className="mt-2 p-4 bg-gray-50 rounded-lg">
                        {getPreviewContent()}
                      </div>
                    </div>
                  </TabsContent>
                </Tabs>

                <div className="flex gap-2 pt-4 border-t">
                  <Button
                    type="submit"
                    disabled={updateSeoMutation.isPending}
                    className="flex-1"
                  >
                    {updateSeoMutation.isPending ? "Kaydediliyor..." : "Kaydet"}
                  </Button>
                </div>
              </form>
            </CardContent>
          </Card>
        </div>

        {/* SEO Analysis */}
        <div>
          <Card>
            <CardHeader>
              <CardTitle className="flex items-center gap-2">
                <AlertCircle className="h-5 w-5" />
                SEO Analizi
              </CardTitle>
              <CardDescription>
                Mevcut SEO ayarlarınızın durumu ve öneriler
              </CardDescription>
            </CardHeader>
            <CardContent className="space-y-4">
              <div className="space-y-3">
                <div className="flex items-center justify-between p-3 bg-green-50 rounded-lg">
                  <div className="flex items-center gap-2">
                    <CheckCircle className="h-4 w-4 text-green-600" />
                    <span className="text-sm">Sayfa Başlığı</span>
                  </div>
                  <span className="text-xs text-green-600">Mevcut</span>
                </div>

                <div className="flex items-center justify-between p-3 bg-green-50 rounded-lg">
                  <div className="flex items-center gap-2">
                    <CheckCircle className="h-4 w-4 text-green-600" />
                    <span className="text-sm">Meta Açıklama</span>
                  </div>
                  <span className="text-xs text-green-600">Mevcut</span>
                </div>

                <div className="flex items-center justify-between p-3 bg-yellow-50 rounded-lg">
                  <div className="flex items-center gap-2">
                    <AlertCircle className="h-4 w-4 text-yellow-600" />
                    <span className="text-sm">OG Görseli</span>
                  </div>
                  <span className="text-xs text-yellow-600">
                    {watchedValues.ogImage ? "Mevcut" : "Eksik"}
                  </span>
                </div>

                <div className="flex items-center justify-between p-3 bg-blue-50 rounded-lg">
                  <div className="flex items-center gap-2">
                    <Globe className="h-4 w-4 text-blue-600" />
                    <span className="text-sm">Yapılandırılmış Veri</span>
                  </div>
                  <span className="text-xs text-blue-600">
                    {watchedValues.structuredData ? "Mevcut" : "Önerilen"}
                  </span>
                </div>
              </div>

              <Separator />

              <div>
                <h4 className="font-medium mb-2">SEO Önerileri</h4>
                <div className="space-y-2 text-sm text-gray-600">
                  <div className="flex items-start gap-2">
                    <div className="w-1.5 h-1.5 bg-blue-600 rounded-full mt-2"></div>
                    <span>Sayfa başlığınızı 50-60 karakter arasında tutun</span>
                  </div>
                  <div className="flex items-start gap-2">
                    <div className="w-1.5 h-1.5 bg-blue-600 rounded-full mt-2"></div>
                    <span>Meta açıklamayı 150-160 karakter arasında yazın</span>
                  </div>
                  <div className="flex items-start gap-2">
                    <div className="w-1.5 h-1.5 bg-blue-600 rounded-full mt-2"></div>
                    <span>Sosyal medya için özel görseller ekleyin</span>
                  </div>
                  <div className="flex items-start gap-2">
                    <div className="w-1.5 h-1.5 bg-blue-600 rounded-full mt-2"></div>
                    <span>Schema.org yapılandırılmış veri kullanın</span>
                  </div>
                </div>
              </div>
            </CardContent>
          </Card>
        </div>
      </div>
    </div>
  );
}