import React, { useState } from "react";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Badge } from "@/components/ui/badge";
import { Textarea } from "@/components/ui/textarea";
import { Dialog, DialogContent, DialogDescription, DialogFooter, DialogHeader, DialogTitle } from "@/components/ui/dialog";
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow } from "@/components/ui/table";
import { useToast } from "@/hooks/use-toast";
import { useMutation, useQuery, useQueryClient } from "@tanstack/react-query";
import { apiRequest } from "@/lib/queryClient";
import { Review } from "@shared/schema";
import { 
  Star, 
  Search, 
  Check, 
  X, 
  Eye,
  MessageSquare,
  Calendar,
  User,
  TrendingUp,
  Award,
  Clock
} from "lucide-react";

export default function ReviewManagement() {
  const { toast } = useToast();
  const queryClient = useQueryClient();
  const [searchTerm, setSearchTerm] = useState("");
  const [selectedReview, setSelectedReview] = useState<Review | null>(null);
  const [filter, setFilter] = useState<"all" | "approved" | "pending">("all");
  
  // Fetch reviews
  const { data: reviews = [], isLoading } = useQuery<Review[]>({
    queryKey: ["/api/reviews"],
  });

  // Approve review mutation
  const approveReviewMutation = useMutation({
    mutationFn: async (id: number) => {
      const response = await apiRequest("PATCH", `/api/reviews/${id}`, { isApproved: true });
      return response.json();
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/reviews"] });
      toast({
        title: "Başarılı",
        description: "Değerlendirme onaylandı",
      });
    },
    onError: (error: Error) => {
      toast({
        title: "Hata",
        description: error.message,
        variant: "destructive",
      });
    },
  });

  // Reject review mutation
  const rejectReviewMutation = useMutation({
    mutationFn: async (id: number) => {
      const response = await apiRequest("PATCH", `/api/reviews/${id}`, { isApproved: false });
      return response.json();
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/reviews"] });
      toast({
        title: "Başarılı",
        description: "Değerlendirme reddedildi",
      });
    },
    onError: (error: Error) => {
      toast({
        title: "Hata",
        description: error.message,
        variant: "destructive",
      });
    },
  });

  // Delete review mutation
  const deleteReviewMutation = useMutation({
    mutationFn: async (id: number) => {
      await apiRequest("DELETE", `/api/reviews/${id}`);
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/reviews"] });
      toast({
        title: "Başarılı",
        description: "Değerlendirme silindi",
      });
    },
    onError: (error: Error) => {
      toast({
        title: "Hata",
        description: error.message,
        variant: "destructive",
      });
    },
  });

  // Mock data for demonstration
  const mockReviews: Review[] = [
    {
      id: 1,
      orderId: 1,
      customerName: "Ahmet Yılmaz",
      customerEmail: "ahmet@example.com",
      rating: 5,
      comment: "Harika bir deneyimdi! Yemekler çok lezzetliydi ve servis hızlıydı.",
      menuItemId: null,
      isApproved: true,
      isPublic: true,
      createdAt: new Date("2024-01-15T10:30:00Z"),
    },
    {
      id: 2,
      orderId: 2,
      customerName: "Fatma Demir",
      customerEmail: "fatma@example.com",
      rating: 4,
      comment: "Genel olarak memnunum ama biraz daha hızlı servis bekliyordum.",
      menuItemId: null,
      isApproved: false,
      isPublic: true,
      createdAt: new Date("2024-01-14T15:45:00Z"),
    },
    {
      id: 3,
      orderId: 3,
      customerName: "Mehmet Öz",
      customerEmail: "mehmet@example.com",
      rating: 5,
      comment: "Kesinlikle tavsiye ederim! Özellikle künefe muhteşemdi.",
      menuItemId: null,
      isApproved: true,
      isPublic: true,
      createdAt: new Date("2024-01-13T20:15:00Z"),
    },
    {
      id: 4,
      orderId: 4,
      customerName: "Ayşe Kaya",
      customerEmail: "ayse@example.com",
      rating: 3,
      comment: "Ortalama bir deneyim. Fiyatlar biraz yüksek gibi.",
      menuItemId: null,
      isApproved: false,
      isPublic: true,
      createdAt: new Date("2024-01-12T12:20:00Z"),
    },
  ];

  const displayReviews = reviews.length > 0 ? reviews : mockReviews;

  const filteredReviews = displayReviews.filter(review => {
    const matchesSearch = review.customerName.toLowerCase().includes(searchTerm.toLowerCase()) ||
                         review.comment?.toLowerCase().includes(searchTerm.toLowerCase());
    
    const matchesFilter = filter === "all" || 
                         (filter === "approved" && review.isApproved) ||
                         (filter === "pending" && !review.isApproved);
    
    return matchesSearch && matchesFilter;
  });

  const getStarRating = (rating: number) => {
    return [...Array(5)].map((_, i) => (
      <Star
        key={i}
        className={`h-4 w-4 ${i < rating ? 'text-yellow-400 fill-current' : 'text-gray-300'}`}
      />
    ));
  };

  const handleApprove = (id: number) => {
    approveReviewMutation.mutate(id);
  };

  const handleReject = (id: number) => {
    rejectReviewMutation.mutate(id);
  };

  const handleDelete = (id: number) => {
    if (confirm("Bu değerlendirmeyi silmek istediğinizden emin misiniz?")) {
      deleteReviewMutation.mutate(id);
    }
  };

  const averageRating = displayReviews.reduce((sum, review) => sum + review.rating, 0) / displayReviews.length;
  const totalReviews = displayReviews.length;
  const approvedReviews = displayReviews.filter(r => r.isApproved).length;
  const pendingReviews = displayReviews.filter(r => !r.isApproved).length;

  return (
    <div className="space-y-6">
      {/* Header */}
      <div className="flex justify-between items-center">
        <div>
          <h2 className="text-2xl font-bold tracking-tight">Değerlendirme Yönetimi</h2>
          <p className="text-muted-foreground">
            Müşteri değerlendirmelerini inceleyin ve yönetin
          </p>
        </div>
      </div>

      {/* Stats Cards */}
      <div className="grid grid-cols-1 md:grid-cols-4 gap-4">
        <Card>
          <CardContent className="p-6">
            <div className="flex items-center justify-between">
              <div>
                <p className="text-sm font-medium text-muted-foreground">Ortalama Puan</p>
                <div className="flex items-center space-x-2">
                  <p className="text-2xl font-bold">{averageRating.toFixed(1)}</p>
                  <div className="flex">
                    {getStarRating(Math.round(averageRating))}
                  </div>
                </div>
              </div>
              <Award className="h-8 w-8 text-yellow-600" />
            </div>
          </CardContent>
        </Card>

        <Card>
          <CardContent className="p-6">
            <div className="flex items-center justify-between">
              <div>
                <p className="text-sm font-medium text-muted-foreground">Toplam Değerlendirme</p>
                <p className="text-2xl font-bold">{totalReviews}</p>
              </div>
              <MessageSquare className="h-8 w-8 text-blue-600" />
            </div>
          </CardContent>
        </Card>

        <Card>
          <CardContent className="p-6">
            <div className="flex items-center justify-between">
              <div>
                <p className="text-sm font-medium text-muted-foreground">Onaylı</p>
                <p className="text-2xl font-bold text-green-600">{approvedReviews}</p>
              </div>
              <Check className="h-8 w-8 text-green-600" />
            </div>
          </CardContent>
        </Card>

        <Card>
          <CardContent className="p-6">
            <div className="flex items-center justify-between">
              <div>
                <p className="text-sm font-medium text-muted-foreground">Bekliyor</p>
                <p className="text-2xl font-bold text-orange-600">{pendingReviews}</p>
              </div>
              <Clock className="h-8 w-8 text-orange-600" />
            </div>
          </CardContent>
        </Card>
      </div>

      {/* Filters and Search */}
      <div className="flex flex-col sm:flex-row gap-4">
        <div className="relative flex-1">
          <Search className="absolute left-3 top-1/2 transform -translate-y-1/2 text-gray-400 h-4 w-4" />
          <Input
            placeholder="Müşteri adı veya yorum ara..."
            value={searchTerm}
            onChange={(e) => setSearchTerm(e.target.value)}
            className="pl-10"
          />
        </div>
        <div className="flex space-x-2">
          <Button
            variant={filter === "all" ? "default" : "outline"}
            onClick={() => setFilter("all")}
          >
            Tümü
          </Button>
          <Button
            variant={filter === "approved" ? "default" : "outline"}
            onClick={() => setFilter("approved")}
          >
            Onaylı
          </Button>
          <Button
            variant={filter === "pending" ? "default" : "outline"}
            onClick={() => setFilter("pending")}
          >
            Bekliyor
          </Button>
        </div>
      </div>

      {/* Reviews Table */}
      <Card>
        <CardHeader>
          <CardTitle className="flex items-center">
            <Star className="h-5 w-5 mr-2" />
            Değerlendirmeler
          </CardTitle>
          <CardDescription>
            {filteredReviews.length} değerlendirme görüntüleniyor
          </CardDescription>
        </CardHeader>
        <CardContent>
          <Table>
            <TableHeader>
              <TableRow>
                <TableHead>Müşteri</TableHead>
                <TableHead>Puan</TableHead>
                <TableHead>Yorum</TableHead>
                <TableHead>Tarih</TableHead>
                <TableHead>Durum</TableHead>
                <TableHead>İşlemler</TableHead>
              </TableRow>
            </TableHeader>
            <TableBody>
              {filteredReviews.map((review) => (
                <TableRow key={review.id}>
                  <TableCell>
                    <div className="flex items-center space-x-2">
                      <User className="h-4 w-4 text-muted-foreground" />
                      <span className="font-medium">{review.customerName}</span>
                    </div>
                  </TableCell>
                  <TableCell>
                    <div className="flex items-center space-x-1">
                      {getStarRating(review.rating)}
                      <span className="ml-2 text-sm font-medium">{review.rating}/5</span>
                    </div>
                  </TableCell>
                  <TableCell>
                    <div className="max-w-xs">
                      <p className="text-sm truncate">{review.comment}</p>
                      {review.comment && review.comment.length > 50 && (
                        <Button
                          variant="ghost"
                          size="sm"
                          onClick={() => setSelectedReview(review)}
                          className="h-auto p-0 text-blue-600"
                        >
                          <Eye className="h-3 w-3 mr-1" />
                          Detay
                        </Button>
                      )}
                    </div>
                  </TableCell>
                  <TableCell>
                    <div className="flex items-center space-x-1">
                      <Calendar className="h-3 w-3 text-muted-foreground" />
                      <span className="text-sm">
                        {review.createdAt ? new Date(review.createdAt).toLocaleDateString('tr-TR') : 'Tarih yok'}
                      </span>
                    </div>
                  </TableCell>
                  <TableCell>
                    <Badge variant={review.isApproved ? "default" : "secondary"}>
                      {review.isApproved ? "Onaylı" : "Bekliyor"}
                    </Badge>
                  </TableCell>
                  <TableCell>
                    <div className="flex space-x-2">
                      {!review.isApproved && (
                        <Button
                          variant="outline"
                          size="sm"
                          onClick={() => handleApprove(review.id)}
                          disabled={approveReviewMutation.isPending}
                        >
                          <Check className="h-4 w-4 text-green-600" />
                        </Button>
                      )}
                      {review.isApproved && (
                        <Button
                          variant="outline"
                          size="sm"
                          onClick={() => handleReject(review.id)}
                          disabled={rejectReviewMutation.isPending}
                        >
                          <X className="h-4 w-4 text-red-600" />
                        </Button>
                      )}
                      <Button
                        variant="outline"
                        size="sm"
                        onClick={() => handleDelete(review.id)}
                        disabled={deleteReviewMutation.isPending}
                      >
                        <X className="h-4 w-4" />
                      </Button>
                    </div>
                  </TableCell>
                </TableRow>
              ))}
            </TableBody>
          </Table>
        </CardContent>
      </Card>

      {/* Review Detail Dialog */}
      <Dialog open={!!selectedReview} onOpenChange={() => setSelectedReview(null)}>
        <DialogContent className="max-w-md">
          <DialogHeader>
            <DialogTitle>Değerlendirme Detayı</DialogTitle>
            <DialogDescription>
              Müşteri değerlendirmesinin tam halini görüntüleyin
            </DialogDescription>
          </DialogHeader>
          {selectedReview && (
            <div className="space-y-4">
              <div>
                <label className="text-sm font-medium">Müşteri</label>
                <p className="text-sm text-muted-foreground">{selectedReview.customerName}</p>
              </div>
              <div>
                <label className="text-sm font-medium">Puan</label>
                <div className="flex items-center space-x-1 mt-1">
                  {getStarRating(selectedReview.rating)}
                  <span className="ml-2 text-sm">{selectedReview.rating}/5</span>
                </div>
              </div>
              <div>
                <label className="text-sm font-medium">Yorum</label>
                <Textarea
                  value={selectedReview.comment || ""}
                  readOnly
                  className="mt-1"
                  rows={4}
                />
              </div>
              <div>
                <label className="text-sm font-medium">Tarih</label>
                <p className="text-sm text-muted-foreground">
                  {selectedReview.createdAt ? new Date(selectedReview.createdAt).toLocaleString('tr-TR') : 'Tarih yok'}
                </p>
              </div>
              <div>
                <label className="text-sm font-medium">Durum</label>
                <div className="mt-1">
                  <Badge variant={selectedReview.isApproved ? "default" : "secondary"}>
                    {selectedReview.isApproved ? "Onaylı" : "Bekliyor"}
                  </Badge>
                </div>
              </div>
            </div>
          )}
          <DialogFooter>
            {selectedReview && !selectedReview.isApproved && (
              <Button
                onClick={() => {
                  handleApprove(selectedReview.id);
                  setSelectedReview(null);
                }}
                disabled={approveReviewMutation.isPending}
              >
                <Check className="h-4 w-4 mr-2" />
                Onayla
              </Button>
            )}
          </DialogFooter>
        </DialogContent>
      </Dialog>

      {/* Rating Distribution */}
      <Card>
        <CardHeader>
          <CardTitle className="flex items-center">
            <TrendingUp className="h-5 w-5 mr-2" />
            Puan Dağılımı
          </CardTitle>
          <CardDescription>
            Müşteri puanlarının dağılımı
          </CardDescription>
        </CardHeader>
        <CardContent>
          <div className="space-y-3">
            {[5, 4, 3, 2, 1].map((rating) => {
              const count = displayReviews.filter(r => r.rating === rating).length;
              const percentage = totalReviews > 0 ? (count / totalReviews) * 100 : 0;
              
              return (
                <div key={rating} className="flex items-center space-x-3">
                  <div className="flex items-center space-x-1 w-12">
                    <span className="text-sm font-medium">{rating}</span>
                    <Star className="h-3 w-3 text-yellow-400 fill-current" />
                  </div>
                  <div className="flex-1 bg-gray-200 rounded-full h-2">
                    <div
                      className="bg-yellow-400 h-2 rounded-full"
                      style={{ width: `${percentage}%` }}
                    ></div>
                  </div>
                  <div className="w-12 text-right">
                    <span className="text-sm text-muted-foreground">{count}</span>
                  </div>
                </div>
              );
            })}
          </div>
        </CardContent>
      </Card>
    </div>
  );
}