import React, { useState } from "react";
import { Restaurant } from "@shared/schema";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Textarea } from "@/components/ui/textarea";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { Switch } from "@/components/ui/switch";
import { Label } from "@/components/ui/label";
import { Separator } from "@/components/ui/separator";
import { useQueryClient } from "@tanstack/react-query";
import { useToast } from "@/hooks/use-toast";
import { apiRequest } from "@/lib/queryClient";
import { Loader2, CreditCard, Banknote, Wallet } from "lucide-react";

interface RestaurantSettingsProps {
  restaurantInfo?: Restaurant;
}

export const RestaurantSettings = ({ restaurantInfo }: RestaurantSettingsProps) => {
  const { toast } = useToast();
  const queryClient = useQueryClient();
  const [isSubmitting, setIsSubmitting] = useState(false);
  const [formData, setFormData] = useState({
    name: restaurantInfo?.name || "",
    address: restaurantInfo?.address || "",
    phone: restaurantInfo?.phone || "",
    email: restaurantInfo?.email || "",
    website: restaurantInfo?.website || "",
    description: restaurantInfo?.description || "",
  });

  // Update form data
  const handleInputChange = (e: React.ChangeEvent<HTMLInputElement | HTMLTextAreaElement>) => {
    const { name, value } = e.target;
    setFormData(prev => ({
      ...prev,
      [name]: value
    }));
  };



  // Update restaurant info
  const handleUpdateRestaurantInfo = async () => {
    setIsSubmitting(true);
    
    try {
      if (restaurantInfo) {
        // Update existing restaurant info
        await apiRequest(`/api/restaurant/${restaurantInfo.id}`, {
          method: "PATCH",
          body: JSON.stringify(formData)
        });
      } else {
        // Create new restaurant info
        await apiRequest("/api/restaurant", {
          method: "POST",
          body: JSON.stringify(formData)
        });
      }

      queryClient.invalidateQueries({ queryKey: ['/api/restaurant'] });
      toast({
        title: "Restoran bilgileri güncellendi",
        description: "Restoran bilgileri başarıyla kaydedildi."
      });
    } catch (error) {
      toast({
        title: "Hata",
        description: "Restoran bilgileri güncellenirken bir hata oluştu.",
        variant: "destructive"
      });
    } finally {
      setIsSubmitting(false);
    }
  };

  return (
    <div className="space-y-6">
      <h2 className="text-2xl font-bold text-primary font-display">Restoran Bilgileri</h2>
      
      <Card>
        <CardHeader>
          <CardTitle>Genel Bilgiler</CardTitle>
          <CardDescription>
            Restoranınızın temel bilgilerini ve iletişim detaylarını güncelleyin.
          </CardDescription>
        </CardHeader>
        <CardContent>
          <div className="space-y-4">
            <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
              <div className="space-y-2">
                <label htmlFor="name">Restoran Adı</label>
                <Input
                  id="name"
                  name="name"
                  value={formData.name}
                  onChange={handleInputChange}
                  placeholder="Örn: Lezzet Köşesi"
                />
              </div>
              <div className="space-y-2">
                <label htmlFor="tagline">Slogan</label>
                <Input
                  id="tagline"
                  name="tagline"
                  value={formData.tagline}
                  onChange={handleInputChange}
                  placeholder="Örn: En iyi yemekler, en iyi hizmet"
                />
              </div>
            </div>
            
            <div className="space-y-2">
              <label htmlFor="address">Adres</label>
              <Textarea
                id="address"
                name="address"
                value={formData.address}
                onChange={handleInputChange}
                placeholder="Örn: Cumhuriyet Mah. Atatürk Cad. No:123, İstanbul"
                rows={2}
              />
            </div>
            
            <div className="space-y-2">
              <label htmlFor="phone">Telefon</label>
              <Input
                id="phone"
                name="phone"
                value={formData.phone}
                onChange={handleInputChange}
                placeholder="Örn: (0212) 123 45 67"
              />
            </div>
            
            <div className="space-y-2">
              <label htmlFor="hoursOfOperation">Çalışma Saatleri</label>
              <Textarea
                id="hoursOfOperation"
                name="hoursOfOperation"
                value={formData.hoursOfOperation}
                onChange={handleInputChange}
                placeholder="Örn: Pazartesi - Cuma: 09:00 - 22:00&#10;Cumartesi - Pazar: 10:00 - 23:00"
                rows={4}
              />
              <p className="text-xs text-gray-500">
                Her bir satıra bir gün veya gün aralığı yazın. Enter tuşu ile satır atlayabilirsiniz.
              </p>
            </div>
            
            <div className="pt-4">
              <Button 
                onClick={handleUpdateRestaurantInfo}
                disabled={isSubmitting}
                className="w-full sm:w-auto"
              >
                {isSubmitting && (
                  <Loader2 className="mr-2 h-4 w-4 animate-spin" />
                )}
                Bilgileri Kaydet
              </Button>
            </div>
          </div>
        </CardContent>
      </Card>

      {/* Payment Settings */}
      <Card>
        <CardHeader>
          <CardTitle className="flex items-center gap-2">
            <CreditCard className="w-5 h-5" />
            Ödeme Sistemi Ayarları
          </CardTitle>
          <CardDescription>
            Müşterilerinizin kullanabileceği ödeme yöntemlerini belirleyin
          </CardDescription>
        </CardHeader>
        <CardContent className="space-y-6">
          {/* Master Payment Switch */}
          <div className="flex items-center justify-between p-4 border rounded-lg">
            <div className="flex items-center gap-3">
              <CreditCard className="w-5 h-5 text-blue-600" />
              <div>
                <Label className="text-base font-medium">Ödeme Sistemi</Label>
                <p className="text-sm text-gray-600">Online sipariş ve ödeme sistemini etkinleştir</p>
              </div>
            </div>
            <Switch
              checked={formData.enablePayments}
              onCheckedChange={(checked) => handleSwitchChange('enablePayments', checked)}
            />
          </div>

          {formData.enablePayments && (
            <>
              <Separator />
              
              {/* Cash Payment */}
              <div className="flex items-center justify-between p-4 border rounded-lg">
                <div className="flex items-center gap-3">
                  <Banknote className="w-5 h-5 text-green-600" />
                  <div>
                    <Label className="text-base font-medium">Nakit Ödeme</Label>
                    <p className="text-sm text-gray-600">Teslimat sırasında nakit ödeme kabul et</p>
                  </div>
                </div>
                <Switch
                  checked={formData.enableCashPayment}
                  onCheckedChange={(checked) => handleSwitchChange('enableCashPayment', checked)}
                />
              </div>

              {/* PayTR */}
              <div className="flex items-center justify-between p-4 border rounded-lg">
                <div className="flex items-center gap-3">
                  <CreditCard className="w-5 h-5 text-blue-600" />
                  <div>
                    <Label className="text-base font-medium">PayTR</Label>
                    <p className="text-sm text-gray-600">Türkiye'nin güvenilir ödeme sistemi</p>
                  </div>
                </div>
                <Switch
                  checked={formData.enablePaytr}
                  onCheckedChange={(checked) => handleSwitchChange('enablePaytr', checked)}
                />
              </div>

              {/* Iyzico */}
              <div className="flex items-center justify-between p-4 border rounded-lg">
                <div className="flex items-center gap-3">
                  <Wallet className="w-5 h-5 text-purple-600" />
                  <div>
                    <Label className="text-base font-medium">Iyzico</Label>
                    <p className="text-sm text-gray-600">Kredi kartı ve dijital cüzdan ödemeleri</p>
                  </div>
                </div>
                <Switch
                  checked={formData.enableIyzico}
                  onCheckedChange={(checked) => handleSwitchChange('enableIyzico', checked)}
                />
              </div>

              {(formData.enablePaytr || formData.enableIyzico) && (
                <div className="bg-yellow-50 border border-yellow-200 rounded-lg p-4">
                  <h4 className="text-sm font-medium text-yellow-800 mb-2">API Anahtarları Gerekli</h4>
                  <p className="text-sm text-yellow-700">
                    Seçtiğiniz ödeme sağlayıcılarının API anahtarlarını sistem yöneticisine bildirmeyi unutmayın.
                    Gerçek ödemeler için üretim API anahtarları gereklidir.
                  </p>
                </div>
              )}
            </>
          )}

          <div className="pt-4">
            <Button 
              onClick={handleUpdateRestaurantInfo}
              disabled={isSubmitting}
              className="w-full sm:w-auto"
            >
              {isSubmitting && (
                <Loader2 className="mr-2 h-4 w-4 animate-spin" />
              )}
              Ödeme Ayarlarını Kaydet
            </Button>
          </div>
        </CardContent>
      </Card>
    </div>
  );
};