import React, { useState, useRef } from "react";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs";
import { Slider } from "@/components/ui/slider";
import { Switch } from "@/components/ui/switch";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { useToast } from "@/hooks/use-toast";
import QRCode from "qrcode";
import { Separator } from "@/components/ui/separator";
import { FileImage, Download, Image, RefreshCw, Link as LinkIcon } from "lucide-react";

export default function QRCodeGenerator() {
  const canvasRef = useRef<HTMLCanvasElement>(null);
  const [url, setUrl] = useState("http://localhost:5000/");
  const [color, setColor] = useState("#000000");
  const [backgroundColor, setBackgroundColor] = useState("#FFFFFF");
  const [size, setSize] = useState<number[]>([200]);
  const [margin, setMargin] = useState<number[]>([4]);
  const [logoEnabled, setLogoEnabled] = useState(false);
  const [logoFile, setLogoFile] = useState<File | null>(null);
  const [logoDataURL, setLogoDataURL] = useState<string | null>(null);
  const [logoSize, setLogoSize] = useState<number[]>([40]);
  const [qrDataURL, setQrDataURL] = useState<string | null>(null);
  const [errorCorrectionLevel, setErrorCorrectionLevel] = useState<"L" | "M" | "Q" | "H">("M");
  const [isGenerating, setIsGenerating] = useState(false);
  
  const { toast } = useToast();
  
  const handleLogoChange = (e: React.ChangeEvent<HTMLInputElement>) => {
    const files = e.target.files;
    if (files && files.length > 0) {
      const file = files[0];
      if (file.type.startsWith("image/")) {
        setLogoFile(file);
        const reader = new FileReader();
        reader.onload = (e) => {
          if (e.target && typeof e.target.result === "string") {
            setLogoDataURL(e.target.result);
          }
        };
        reader.readAsDataURL(file);
      } else {
        toast({
          title: "Geçersiz dosya formatı",
          description: "Lütfen geçerli bir görsel dosyası seçin (PNG, JPG, JPEG, GIF, SVG).",
          variant: "destructive",
        });
      }
    }
  };
  
  const generateQRCode = async () => {
    if (canvasRef.current) {
      try {
        setIsGenerating(true);
        
        // QR kodu oluştur
        const qrOptions = {
          errorCorrectionLevel,
          margin: margin[0],
          width: size[0],
          color: {
            dark: color,
            light: backgroundColor,
          },
        };
        
        // Canvas'a QR kodu çiz
        await QRCode.toCanvas(canvasRef.current, url, qrOptions);
        
        // QR kodunu DataURL olarak sakla
        const qrDataURL = canvasRef.current.toDataURL("image/png");
        setQrDataURL(qrDataURL);
        
        // Logo eklenmişse ve etkinse logo ekle
        if (logoEnabled && logoDataURL) {
          const ctx = canvasRef.current.getContext("2d");
          if (ctx) {
            // Logo için görsel oluştur
            const logo = new Image();
            logo.src = logoDataURL as string;
            
            await new Promise<void>((resolve) => {
              logo.onload = () => {
                // Logonun yerleştirileceği konum (merkezde olacak şekilde)
                const logoWidth = logoSize[0];
                const logoHeight = logoWidth; // Kare logo için
                const x = (size[0] - logoWidth) / 2;
                const y = (size[0] - logoHeight) / 2;
                
                // Logo için arka plan beyaz alan oluştur
                ctx.fillStyle = "white";
                ctx.fillRect(x - 5, y - 5, logoWidth + 10, logoHeight + 10);
                
                // Logoyu çiz
                ctx.drawImage(logo, x, y, logoWidth, logoHeight);
                
                // Son halini DataURL olarak kaydet
                const finalQRDataURL = canvasRef.current!.toDataURL("image/png");
                setQrDataURL(finalQRDataURL);
                resolve();
              };
            });
          }
        }
        
        toast({
          title: "QR Kod Oluşturuldu",
          description: "QR kod başarıyla oluşturuldu. İndirmek için butona tıklayın.",
        });
      } catch (error) {
        console.error("QR kod oluşturma hatası:", error);
        toast({
          title: "QR Kod Oluşturma Hatası",
          description: "QR kod oluşturulurken bir hata oluştu. Lütfen tekrar deneyin.",
          variant: "destructive",
        });
      } finally {
        setIsGenerating(false);
      }
    }
  };
  
  const handleDownload = () => {
    if (qrDataURL) {
      const a = document.createElement("a");
      a.href = qrDataURL;
      a.download = "jouska-menu-qrcode.png";
      document.body.appendChild(a);
      a.click();
      document.body.removeChild(a);
      
      toast({
        title: "QR Kod İndiriliyor",
        description: "QR kod indirme işlemi başlatıldı.",
      });
    }
  };
  
  return (
    <Card className="w-full">
      <CardHeader>
        <CardTitle>QR Kod Oluşturucu</CardTitle>
        <CardDescription>
          Menünüz için özelleştirilmiş QR kodları oluşturun ve kullanın.
        </CardDescription>
      </CardHeader>
      <CardContent>
        <Tabs defaultValue="generator">
          <TabsList className="grid w-full grid-cols-2">
            <TabsTrigger value="generator">QR Kod Oluştur</TabsTrigger>
            <TabsTrigger value="designer">QR Kod Tasarımı</TabsTrigger>
          </TabsList>
          
          <TabsContent value="generator" className="mt-4 space-y-4">
            <div className="grid grid-cols-1 lg:grid-cols-2 gap-6">
              <div className="space-y-4">
                <div className="space-y-2">
                  <Label htmlFor="url">URL</Label>
                  <div className="flex gap-2">
                    <Input
                      id="url"
                      placeholder="https://menu.jouskarestaurant.com"
                      value={url}
                      onChange={(e) => setUrl(e.target.value)}
                    />
                  </div>
                  <p className="text-xs text-muted-foreground">
                    QR kodunuzun yönlendireceği URL adresi.
                  </p>
                </div>
                
                <div className="space-y-2">
                  <Label htmlFor="size">Boyut</Label>
                  <Slider
                    id="size"
                    min={100}
                    max={500}
                    step={10}
                    value={size}
                    onValueChange={setSize}
                  />
                  <div className="flex justify-between text-xs text-muted-foreground">
                    <span>100px</span>
                    <span>{size[0]}px</span>
                    <span>500px</span>
                  </div>
                </div>
                
                <div className="space-y-2">
                  <Label htmlFor="margin">Kenar Boşluğu</Label>
                  <Slider
                    id="margin"
                    min={0}
                    max={10}
                    step={1}
                    value={margin}
                    onValueChange={setMargin}
                  />
                  <div className="flex justify-between text-xs text-muted-foreground">
                    <span>0</span>
                    <span>{margin[0]}</span>
                    <span>10</span>
                  </div>
                </div>
                
                <div className="space-y-2">
                  <Label htmlFor="errorCorrectionLevel">Hata Düzeltme Seviyesi</Label>
                  <Select 
                    value={errorCorrectionLevel} 
                    onValueChange={(value) => setErrorCorrectionLevel(value as "L" | "M" | "Q" | "H")}
                  >
                    <SelectTrigger>
                      <SelectValue placeholder="Seviye Seçin" />
                    </SelectTrigger>
                    <SelectContent>
                      <SelectItem value="L">Düşük (L - %7 onarım)</SelectItem>
                      <SelectItem value="M">Orta (M - %15 onarım)</SelectItem>
                      <SelectItem value="Q">Yüksek (Q - %25 onarım)</SelectItem>
                      <SelectItem value="H">En Yüksek (H - %30 onarım)</SelectItem>
                    </SelectContent>
                  </Select>
                  <p className="text-xs text-muted-foreground">
                    Yüksek hata düzeltme, QR koduna logo eklediğinizde taranabilirliği artırır.
                  </p>
                </div>
                
                <div className="flex items-center justify-center mt-4">
                  <Button onClick={generateQRCode} className="w-full" disabled={isGenerating}>
                    {isGenerating ? (
                      <>
                        <RefreshCw className="h-4 w-4 mr-2 animate-spin" />
                        Oluşturuluyor...
                      </>
                    ) : (
                      <>
                        <LinkIcon className="h-4 w-4 mr-2" />
                        QR Kod Oluştur
                      </>
                    )}
                  </Button>
                </div>
              </div>
              
              <div className="flex flex-col items-center justify-center">
                <div className="flex flex-col items-center space-y-4">
                  <div 
                    className="flex items-center justify-center border rounded-md shadow-sm p-4 bg-white"
                    style={{ width: `${Math.max(size[0], 200)}px`, height: `${Math.max(size[0], 200)}px` }}
                  >
                    <canvas ref={canvasRef} className="max-w-full max-h-full" />
                  </div>
                  
                  {qrDataURL && (
                    <Button variant="outline" onClick={handleDownload}>
                      <Download className="h-4 w-4 mr-2" />
                      QR Kodu İndir
                    </Button>
                  )}
                </div>
              </div>
            </div>
          </TabsContent>
          
          <TabsContent value="designer" className="mt-4 space-y-4">
            <div className="grid grid-cols-1 lg:grid-cols-2 gap-6">
              <div className="space-y-4">
                <div className="space-y-2">
                  <Label htmlFor="color">QR Kod Rengi</Label>
                  <div className="flex gap-2">
                    <Input
                      id="color"
                      type="color"
                      value={color}
                      onChange={(e) => setColor(e.target.value)}
                      className="w-12"
                    />
                    <Input
                      value={color}
                      onChange={(e) => setColor(e.target.value)}
                      className="flex-1"
                    />
                  </div>
                </div>
                
                <div className="space-y-2">
                  <Label htmlFor="backgroundColor">Arka Plan Rengi</Label>
                  <div className="flex gap-2">
                    <Input
                      id="backgroundColor"
                      type="color"
                      value={backgroundColor}
                      onChange={(e) => setBackgroundColor(e.target.value)}
                      className="w-12"
                    />
                    <Input
                      value={backgroundColor}
                      onChange={(e) => setBackgroundColor(e.target.value)}
                      className="flex-1"
                    />
                  </div>
                </div>
                
                <Separator className="my-4" />
                
                <div className="space-y-2">
                  <div className="flex items-center justify-between">
                    <Label htmlFor="logoEnabled">Logo Kullan</Label>
                    <Switch
                      id="logoEnabled"
                      checked={logoEnabled}
                      onCheckedChange={setLogoEnabled}
                    />
                  </div>
                  <p className="text-xs text-muted-foreground">
                    QR kodunuzun ortasına restoranınızın logosu eklenecektir.
                  </p>
                </div>
                
                {logoEnabled && (
                  <>
                    <div className="space-y-2">
                      <Label htmlFor="logoFile">Logo Dosyası</Label>
                      <div className="flex items-center gap-2">
                        <Input
                          id="logoFile"
                          type="file"
                          accept="image/*"
                          onChange={handleLogoChange}
                          className="flex-1"
                        />
                      </div>
                      <p className="text-xs text-muted-foreground">
                        En iyi sonuç için şeffaf arkaplanı olan kare logo kullanın.
                      </p>
                    </div>
                    
                    <div className="space-y-2">
                      <Label htmlFor="logoSize">Logo Boyutu</Label>
                      <Slider
                        id="logoSize"
                        min={20}
                        max={100}
                        step={5}
                        value={logoSize}
                        onValueChange={setLogoSize}
                        disabled={!logoEnabled || !logoFile}
                      />
                      <div className="flex justify-between text-xs text-muted-foreground">
                        <span>Küçük</span>
                        <span>{logoSize[0]}px</span>
                        <span>Büyük</span>
                      </div>
                    </div>
                    
                    {logoDataURL && (
                      <div className="flex justify-center py-2">
                        <div className="relative w-24 h-24 border rounded overflow-hidden">
                          <img 
                            src={logoDataURL} 
                            alt="Logo Önizleme" 
                            className="w-full h-full object-contain"
                          />
                        </div>
                      </div>
                    )}
                  </>
                )}
              </div>
              
              <div className="flex flex-col items-center justify-center">
                <div className="flex flex-col items-center space-y-4">
                  <div 
                    className="flex items-center justify-center border rounded-md shadow-sm p-4 bg-white"
                    style={{ width: `${Math.max(size[0], 200)}px`, height: `${Math.max(size[0], 200)}px` }}
                  >
                    <canvas ref={canvasRef} className="max-w-full max-h-full" />
                  </div>
                  
                  <div className="flex gap-2">
                    <Button 
                      onClick={generateQRCode} 
                      disabled={isGenerating}
                    >
                      {isGenerating ? (
                        <>
                          <RefreshCw className="h-4 w-4 mr-2 animate-spin" />
                          Oluşturuluyor...
                        </>
                      ) : (
                        <>
                          <RefreshCw className="h-4 w-4 mr-2" />
                          Önizleme Oluştur
                        </>
                      )}
                    </Button>
                    
                    {qrDataURL && (
                      <Button variant="outline" onClick={handleDownload}>
                        <Download className="h-4 w-4 mr-2" />
                        QR Kodu İndir
                      </Button>
                    )}
                  </div>
                </div>
              </div>
            </div>
          </TabsContent>
        </Tabs>
      </CardContent>
    </Card>
  );
}