import React from "react";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Switch } from "@/components/ui/switch";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Separator } from "@/components/ui/separator";
import { Badge } from "@/components/ui/badge";
import { useToast } from "@/hooks/use-toast";
import { useMutation, useQuery, useQueryClient } from "@tanstack/react-query";
import { apiRequest } from "@/lib/queryClient";
import { Restaurant } from "@shared/schema";
import { 
  CreditCard, 
  DollarSign, 
  Settings, 
  CheckCircle, 
  XCircle,
  AlertTriangle,
  Wallet
} from "lucide-react";

interface PaymentSettingsProps {
  restaurantInfo?: Restaurant;
}

export function PaymentSettings({ restaurantInfo }: PaymentSettingsProps) {
  const { toast } = useToast();
  const queryClient = useQueryClient();
  const [formData, setFormData] = React.useState({
    enablePayments: restaurantInfo?.enablePayments || false,
    enablePaytr: restaurantInfo?.enablePaytr || false,
    enableIyzico: restaurantInfo?.enableIyzico || false,
    enableCashPayment: restaurantInfo?.enableCashPayment || true,
    paytrCommissionRate: restaurantInfo?.paytrCommissionRate || "2.9",
    iyzicoCommissionRate: restaurantInfo?.iyzicoCommissionRate || "2.7",
    fixedFeePaytr: restaurantInfo?.fixedFeePaytr || "0.25",
    fixedFeeIyzico: restaurantInfo?.fixedFeeIyzico || "0.30",
  });

  const updateRestaurantMutation = useMutation({
    mutationFn: async (data: Partial<Restaurant>) => {
      const restaurantId = restaurantInfo?.id || 2; // Use current restaurant ID
      const response = await apiRequest("PATCH", `/api/restaurant/${restaurantId}`, data);
      return response.json();
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/restaurant"] });
      toast({
        title: "Başarılı",
        description: "Ödeme sistemi ayarları güncellendi",
      });
    },
    onError: (error: Error) => {
      toast({
        title: "Hata",
        description: error.message,
        variant: "destructive",
      });
    },
  });

  const handleInputChange = (field: string, value: string | boolean) => {
    setFormData(prev => ({
      ...prev,
      [field]: value
    }));
  };

  const handleSave = () => {
    updateRestaurantMutation.mutate(formData);
  };

  const handlePaymentToggle = async (field: string, enabled: boolean) => {
    await updateRestaurantMutation.mutateAsync({
      [field]: enabled
    });
  };

  const getPaymentStatus = () => {
    if (!restaurantInfo) return { status: "loading", count: 0 };
    
    const paymentMethods = [
      restaurantInfo.enablePaytr,
      restaurantInfo.enableIyzico,
      restaurantInfo.enableCashPayment
    ].filter(Boolean);

    if (paymentMethods.length === 0) {
      return { status: "disabled", count: 0 };
    } else if (paymentMethods.length === 1) {
      return { status: "partial", count: paymentMethods.length };
    } else {
      return { status: "active", count: paymentMethods.length };
    }
  };

  const paymentStatus = getPaymentStatus();

  return (
    <div className="space-y-6">
      {/* Genel Durum Kartı */}
      <Card>
        <CardHeader>
          <div className="flex items-center justify-between">
            <div className="flex items-center space-x-2">
              <CreditCard className="h-5 w-5 text-primary" />
              <CardTitle>Ödeme Sistemi Durumu</CardTitle>
            </div>
            <div className="flex items-center space-x-2">
              {paymentStatus.status === "active" && (
                <Badge variant="default" className="bg-green-500">
                  <CheckCircle className="h-3 w-3 mr-1" />
                  Aktif
                </Badge>
              )}
              {paymentStatus.status === "partial" && (
                <Badge variant="secondary" className="bg-yellow-500">
                  <AlertTriangle className="h-3 w-3 mr-1" />
                  Kısmi
                </Badge>
              )}
              {paymentStatus.status === "disabled" && (
                <Badge variant="destructive">
                  <XCircle className="h-3 w-3 mr-1" />
                  Kapalı
                </Badge>
              )}
            </div>
          </div>
          <CardDescription>
            {paymentStatus.count > 0 
              ? `${paymentStatus.count} ödeme yöntemi aktif`
              : "Hiç ödeme yöntemi aktif değil"
            }
          </CardDescription>
        </CardHeader>
        <CardContent className="space-y-4">
          <div className="flex items-center space-x-4">
            <Switch
              id="master-payment-toggle"
              checked={restaurantInfo?.enablePayments || false}
              onCheckedChange={(checked) => handlePaymentToggle("enablePayments", checked)}
              disabled={updateRestaurantMutation.isPending}
            />
            <div className="flex-1">
              <Label htmlFor="master-payment-toggle" className="text-base font-medium">
                Ana Ödeme Sistemi
              </Label>
              <p className="text-sm text-muted-foreground">
                Tüm ödeme sistemlerini ana kontrol anahtarı
              </p>
            </div>
          </div>

          <Separator />

          <div className="flex items-center space-x-4">
            <Switch
              id="order-system-toggle"
              checked={restaurantInfo?.enableOrderSystem || false}
              onCheckedChange={(checked) => handlePaymentToggle("enableOrderSystem", checked)}
              disabled={updateRestaurantMutation.isPending}
            />
            <div className="flex-1">
              <Label htmlFor="order-system-toggle" className="text-base font-medium">
                Sipariş Sistemi
              </Label>
              <p className="text-sm text-muted-foreground">
                Müşteriler menü kartlarında sipariş butonlarını görebilir
              </p>
            </div>
          </div>
        </CardContent>
      </Card>

      {/* Ödeme Yöntemleri */}
      <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-4">
        
        {/* PayTR */}
        <Card>
          <CardHeader className="pb-3">
            <div className="flex items-center justify-between">
              <CardTitle className="text-lg flex items-center">
                <CreditCard className="h-4 w-4 mr-2" />
                PayTR
              </CardTitle>
              <Switch
                checked={restaurantInfo?.enablePaytr || false}
                onCheckedChange={(checked) => handlePaymentToggle("enablePaytr", checked)}
                disabled={updateRestaurantMutation.isPending || !restaurantInfo?.enablePayments}
              />
            </div>
            <CardDescription>
              Türkiye'nin önde gelen ödeme sistemleri
            </CardDescription>
          </CardHeader>
          <CardContent>
            <div className="space-y-3">
              <div className="flex justify-between text-sm">
                <span>Durumu:</span>
                <span className={restaurantInfo?.enablePaytr ? "text-green-600" : "text-gray-500"}>
                  {restaurantInfo?.enablePaytr ? "Aktif" : "Kapalı"}
                </span>
              </div>
              <div className="flex justify-between text-sm">
                <span>Kart Ödemeleri:</span>
                <span>✓ Destekleniyor</span>
              </div>
              <div className="space-y-2">
                <Label htmlFor="paytr-commission">Komisyon Oranı (%)</Label>
                <Input
                  id="paytr-commission"
                  type="number"
                  step="0.1"
                  min="0"
                  max="10"
                  value={formData.paytrCommissionRate}
                  onChange={(e) => handleInputChange('paytrCommissionRate', e.target.value)}
                  placeholder="2.9"
                />
              </div>
              <div className="space-y-2">
                <Label htmlFor="paytr-fixed-fee">Sabit Ücret (TL)</Label>
                <Input
                  id="paytr-fixed-fee"
                  type="number"
                  step="0.01"
                  min="0"
                  value={formData.fixedFeePaytr}
                  onChange={(e) => handleInputChange('fixedFeePaytr', e.target.value)}
                  placeholder="0.25"
                />
              </div>
            </div>
          </CardContent>
        </Card>

        {/* Iyzico */}
        <Card>
          <CardHeader className="pb-3">
            <div className="flex items-center justify-between">
              <CardTitle className="text-lg flex items-center">
                <DollarSign className="h-4 w-4 mr-2" />
                Iyzico
              </CardTitle>
              <Switch
                checked={restaurantInfo?.enableIyzico || false}
                onCheckedChange={(checked) => handlePaymentToggle("enableIyzico", checked)}
                disabled={updateRestaurantMutation.isPending || !restaurantInfo?.enablePayments}
              />
            </div>
            <CardDescription>
              Güvenli online ödeme çözümleri
            </CardDescription>
          </CardHeader>
          <CardContent>
            <div className="space-y-3">
              <div className="flex justify-between text-sm">
                <span>Durumu:</span>
                <span className={restaurantInfo?.enableIyzico ? "text-green-600" : "text-gray-500"}>
                  {restaurantInfo?.enableIyzico ? "Aktif" : "Kapalı"}
                </span>
              </div>
              <div className="flex justify-between text-sm">
                <span>Kart Ödemeleri:</span>
                <span>✓ Destekleniyor</span>
              </div>
              <div className="space-y-2">
                <Label htmlFor="iyzico-commission">Komisyon Oranı (%)</Label>
                <Input
                  id="iyzico-commission"
                  type="number"
                  step="0.1"
                  min="0"
                  max="10"
                  value={formData.iyzicoCommissionRate}
                  onChange={(e) => handleInputChange('iyzicoCommissionRate', e.target.value)}
                  placeholder="2.7"
                />
              </div>
              <div className="space-y-2">
                <Label htmlFor="iyzico-fixed-fee">Sabit Ücret (TL)</Label>
                <Input
                  id="iyzico-fixed-fee"
                  type="number"
                  step="0.01"
                  min="0"
                  value={formData.fixedFeeIyzico}
                  onChange={(e) => handleInputChange('fixedFeeIyzico', e.target.value)}
                  placeholder="0.30"
                />
              </div>
            </div>
          </CardContent>
        </Card>

        {/* Nakit Ödeme */}
        <Card>
          <CardHeader className="pb-3">
            <div className="flex items-center justify-between">
              <CardTitle className="text-lg flex items-center">
                <Wallet className="h-4 w-4 mr-2" />
                Nakit Ödeme
              </CardTitle>
              <Switch
                checked={restaurantInfo?.enableCashPayment || false}
                onCheckedChange={(checked) => handlePaymentToggle("enableCashPayment", checked)}
                disabled={updateRestaurantMutation.isPending || !restaurantInfo?.enablePayments}
              />
            </div>
            <CardDescription>
              Kapıda nakit ödeme seçeneği
            </CardDescription>
          </CardHeader>
          <CardContent>
            <div className="space-y-3">
              <div className="flex justify-between text-sm">
                <span>Durumu:</span>
                <span className={restaurantInfo?.enableCashPayment ? "text-green-600" : "text-gray-500"}>
                  {restaurantInfo?.enableCashPayment ? "Aktif" : "Kapalı"}
                </span>
              </div>
              <div className="flex justify-between text-sm">
                <span>Komisyon:</span>
                <span>%0 - Ücretsiz</span>
              </div>
              <div className="flex justify-between text-sm">
                <span>Teslimat:</span>
                <span>Kapıda ödeme</span>
              </div>
            </div>
          </CardContent>
        </Card>
      </div>

      {/* API Ayarları */}
      <Card>
        <CardHeader>
          <CardTitle className="flex items-center">
            <Settings className="h-4 w-4 mr-2" />
            API Ayarları ve Entegrasyon
          </CardTitle>
          <CardDescription>
            Ödeme sağlayıcılarınızın API bilgilerini yönetin
          </CardDescription>
        </CardHeader>
        <CardContent className="space-y-6">
          
          <div className="space-y-4">
            <h4 className="font-medium text-sm text-muted-foreground">PayTR API Bilgileri</h4>
            <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
              <div className="space-y-2">
                <Label htmlFor="paytr-merchant-id">Merchant ID</Label>
                <Input 
                  id="paytr-merchant-id"
                  placeholder="PayTR Merchant ID"
                  type="password"
                  disabled
                />
              </div>
              <div className="space-y-2">
                <Label htmlFor="paytr-merchant-key">Merchant Key</Label>
                <Input 
                  id="paytr-merchant-key"
                  placeholder="PayTR Merchant Key"
                  type="password"
                  disabled
                />
              </div>
            </div>
          </div>

          <Separator />

          <div className="space-y-4">
            <h4 className="font-medium text-sm text-muted-foreground">Iyzico API Bilgileri</h4>
            <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
              <div className="space-y-2">
                <Label htmlFor="iyzico-api-key">API Key</Label>
                <Input 
                  id="iyzico-api-key"
                  placeholder="Iyzico API Key"
                  type="password"
                  disabled
                />
              </div>
              <div className="space-y-2">
                <Label htmlFor="iyzico-secret-key">Secret Key</Label>
                <Input 
                  id="iyzico-secret-key"
                  placeholder="Iyzico Secret Key"
                  type="password"
                  disabled
                />
              </div>
            </div>
          </div>

          <div className="bg-yellow-50 border border-yellow-200 rounded-lg p-4">
            <div className="flex items-start space-x-3">
              <AlertTriangle className="h-5 w-5 text-yellow-600 mt-0.5" />
              <div className="text-sm">
                <p className="font-medium text-yellow-800">API Anahtarları Gerekli</p>
                <p className="text-yellow-700 mt-1">
                  Ödeme sistemlerinin çalışması için API anahtarlarının sunucu ortam değişkenlerinde tanımlanması gerekir.
                  Lütfen sistem yöneticinize başvurun.
                </p>
              </div>
            </div>
          </div>
        </CardContent>
      </Card>

      {/* Test Modu Uyarısı */}
      <Card className="border-orange-200 bg-orange-50">
        <CardHeader>
          <CardTitle className="text-orange-800 flex items-center">
            <AlertTriangle className="h-4 w-4 mr-2" />
            Test Modu Aktif
          </CardTitle>
        </CardHeader>
        <CardContent>
          <p className="text-sm text-orange-700">
            Sistem şu anda test modunda çalışıyor. Gerçek ödemeler için API anahtarlarını ekleyip 
            üretim moduna geçiş yapmanız gerekir.
          </p>
        </CardContent>
      </Card>

      {/* Komisyon Ayarları Kaydet */}
      <Card>
        <CardHeader>
          <CardTitle className="flex items-center">
            <Settings className="h-4 w-4 mr-2" />
            Komisyon Oranları Yönetimi
          </CardTitle>
          <CardDescription>
            PayTR ve Iyzico komisyon oranlarını buradan ayarlayabilirsiniz
          </CardDescription>
        </CardHeader>
        <CardContent className="space-y-4">
          <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
            <div className="space-y-4">
              <h4 className="font-medium text-sm text-muted-foreground">PayTR Komisyon Ayarları</h4>
              <div className="space-y-3">
                <div className="space-y-2">
                  <Label htmlFor="paytr-commission-main">Komisyon Oranı (%)</Label>
                  <Input
                    id="paytr-commission-main"
                    type="number"
                    step="0.1"
                    min="0"
                    max="10"
                    value={formData.paytrCommissionRate}
                    onChange={(e) => handleInputChange('paytrCommissionRate', e.target.value)}
                    placeholder="2.9"
                  />
                </div>
                <div className="space-y-2">
                  <Label htmlFor="paytr-fixed-fee-main">Sabit Ücret (TL)</Label>
                  <Input
                    id="paytr-fixed-fee-main"
                    type="number"
                    step="0.01"
                    min="0"
                    value={formData.fixedFeePaytr}
                    onChange={(e) => handleInputChange('fixedFeePaytr', e.target.value)}
                    placeholder="0.25"
                  />
                </div>
              </div>
            </div>

            <div className="space-y-4">
              <h4 className="font-medium text-sm text-muted-foreground">Iyzico Komisyon Ayarları</h4>
              <div className="space-y-3">
                <div className="space-y-2">
                  <Label htmlFor="iyzico-commission-main">Komisyon Oranı (%)</Label>
                  <Input
                    id="iyzico-commission-main"
                    type="number"
                    step="0.1"
                    min="0"
                    max="10"
                    value={formData.iyzicoCommissionRate}
                    onChange={(e) => handleInputChange('iyzicoCommissionRate', e.target.value)}
                    placeholder="2.7"
                  />
                </div>
                <div className="space-y-2">
                  <Label htmlFor="iyzico-fixed-fee-main">Sabit Ücret (TL)</Label>
                  <Input
                    id="iyzico-fixed-fee-main"
                    type="number"
                    step="0.01"
                    min="0"
                    value={formData.fixedFeeIyzico}
                    onChange={(e) => handleInputChange('fixedFeeIyzico', e.target.value)}
                    placeholder="0.30"
                  />
                </div>
              </div>
            </div>
          </div>

          <div className="flex justify-end space-x-3 pt-4">
            <Button
              onClick={handleSave}
              disabled={updateRestaurantMutation.isPending}
              className="bg-primary hover:bg-primary/90"
            >
              {updateRestaurantMutation.isPending ? "Kaydediliyor..." : "Komisyon Ayarlarını Kaydet"}
            </Button>
          </div>

          <div className="bg-blue-50 border border-blue-200 rounded-lg p-4 mt-4">
            <div className="flex items-start space-x-3">
              <DollarSign className="h-5 w-5 text-blue-600 mt-0.5" />
              <div className="text-sm">
                <p className="font-medium text-blue-800">Komisyon Hesaplama</p>
                <p className="text-blue-700 mt-1">
                  Her işlem için: (Tutar × Komisyon Oranı / 100) + Sabit Ücret şeklinde hesaplanır.
                  <br />
                  Örnek: 100 TL için PayTR = (100 × 2.9 / 100) + 0.25 = 3.15 TL komisyon
                </p>
              </div>
            </div>
          </div>
        </CardContent>
      </Card>
    </div>
  );
}