import React, { useState } from "react";
import {
  Card,
  CardContent,
  CardDescription,
  CardFooter,
  CardHeader,
  CardTitle,
} from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Textarea } from "@/components/ui/textarea";
import { 
  Alert, 
  AlertDescription, 
  AlertTitle 
} from "@/components/ui/alert";
import { Loader2, Save, Upload, AlertTriangle } from "lucide-react";
import { useMutation } from "@tanstack/react-query";
import { queryClient } from "@/lib/queryClient";
import { toast } from "@/hooks/use-toast";
import { useCategories } from "@/hooks/use-categories";

export function JouskaMenuImporter() {
  const [inputText, setInputText] = useState("");
  const [parsedItems, setParsedItems] = useState<any[]>([]);
  const [isParsing, setIsParsing] = useState(false);
  const [selectedCategory, setSelectedCategory] = useState<number | null>(null);
  const { data: categories, isLoading: categoriesLoading } = useCategories();

  // Parse the input text to extract menu items
  const parseMenuItems = () => {
    setIsParsing(true);
    
    try {
      // Simple parsing logic - adjust based on your input format
      const lines = inputText.split("\n").filter(line => line.trim());
      
      const items = [];
      let currentItem: any = {};
      
      for (let i = 0; i < lines.length; i++) {
        const line = lines[i].trim();
        
        // Check if it's a price line (ends with TL, ₺, or just numbers)
        if (/\d+(\.\d+)?\s*(TL|₺|\s*)$/.test(line)) {
          // Extract price
          const price = line.match(/\d+(\.\d+)?/)?.[0] || "";
          
          // If we already have an item name, complete this item
          if (currentItem.name) {
            currentItem.price = price + " ₺";
            items.push({...currentItem});
            currentItem = {};
          }
        } 
        // If it's not a price and we don't have a name yet, it's an item name
        else if (!currentItem.name) {
          currentItem.name = line;
          currentItem.description = "";
        } 
        // Otherwise it's a description
        else {
          currentItem.description += currentItem.description ? " " + line : line;
        }
      }
      
      // Add last item if exists
      if (currentItem.name && currentItem.price) {
        items.push(currentItem);
      }
      
      setParsedItems(items);
      
      toast({
        title: "Menü öğeleri ayrıştırıldı",
        description: `${items.length} öğe başarıyla ayrıştırıldı.`,
      });
    } catch (error) {
      console.error("Parsing error:", error);
      toast({
        title: "Ayrıştırma hatası",
        description: "Menü öğeleri ayrıştırılırken bir hata oluştu.",
        variant: "destructive",
      });
    } finally {
      setIsParsing(false);
    }
  };

  // Create menu items mutation
  const createMenuItemsMutation = useMutation({
    mutationFn: async (items: any[]) => {
      if (!selectedCategory) {
        throw new Error("Kategori seçilmedi");
      }
      
      const responses = [];
      for (const item of items) {
        const menuItem = {
          name: item.name,
          description: item.description || "",
          price: item.price,
          categoryId: selectedCategory,
          isAvailable: true,
          order: 0 // Bu değer otomatik olarak sıralanacak
        };
        
        const res = await fetch("/api/menu-items", {
          method: "POST",
          headers: { "Content-Type": "application/json" },
          body: JSON.stringify(menuItem),
          credentials: "include"
        });
        
        if (!res.ok) {
          throw new Error(`Menü öğesi eklenirken hata: ${res.statusText}`);
        }
        
        responses.push(await res.json());
      }
      
      return responses;
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ['/api/menu-items'] });
      toast({
        title: "Menü öğeleri eklendi",
        description: `${parsedItems.length} öğe başarıyla eklendi.`,
      });
      // Clear the parsed items
      setParsedItems([]);
      setInputText("");
    },
    onError: (error: Error) => {
      toast({
        title: "Menü öğeleri eklenemedi",
        description: error.message,
        variant: "destructive",
      });
    }
  });

  return (
    <Card className="w-full">
      <CardHeader>
        <CardTitle>Jouska Menü İçeriği Ekle</CardTitle>
        <CardDescription>
          Jouska menüsünden kopyaladığınız metni buraya yapıştırın. Sistem otomatik olarak
          menü öğelerini ve fiyatlarını tespit edecektir.
        </CardDescription>
      </CardHeader>
      
      <CardContent className="space-y-4">
        {parsedItems.length > 0 ? (
          <>
            <div className="flex items-center gap-4 mb-4">
              <label className="font-medium">Kategori seçin:</label>
              <select 
                className="p-2 border rounded-md w-64"
                value={selectedCategory || ""}
                onChange={(e) => setSelectedCategory(Number(e.target.value))}
              >
                <option value="">Kategori seçin</option>
                {categories?.map((category) => (
                  <option key={category.id} value={category.id}>
                    {category.name}
                  </option>
                ))}
              </select>
            </div>
            
            <div className="border rounded-md p-4 bg-gray-50 max-h-[400px] overflow-y-auto">
              <h3 className="font-bold mb-2">Ayrıştırılan Öğeler ({parsedItems.length})</h3>
              <div className="space-y-4">
                {parsedItems.map((item, index) => (
                  <div key={index} className="border-b pb-2">
                    <div className="flex justify-between">
                      <h4 className="font-semibold">{item.name}</h4>
                      <span className="font-bold text-primary">{item.price}</span>
                    </div>
                    {item.description && (
                      <p className="text-sm text-gray-600">{item.description}</p>
                    )}
                  </div>
                ))}
              </div>
            </div>
            
            <Alert className="bg-blue-50">
              <AlertTriangle className="h-4 w-4" />
              <AlertTitle>Kontrol edin</AlertTitle>
              <AlertDescription>
                Menü öğelerini eklemeden önce kategori seçtiğinizden ve öğelerin doğru ayrıştırıldığından emin olun.
              </AlertDescription>
            </Alert>
          </>
        ) : (
          <Textarea
            placeholder="Menü içeriğini buraya yapıştırın..."
            value={inputText}
            onChange={(e) => setInputText(e.target.value)}
            rows={10}
            className="font-mono text-sm"
          />
        )}
      </CardContent>
      
      <CardFooter className="flex justify-end gap-2">
        {parsedItems.length > 0 ? (
          <>
            <Button 
              variant="outline" 
              onClick={() => setParsedItems([])}
            >
              İptal
            </Button>
            <Button 
              onClick={() => createMenuItemsMutation.mutate(parsedItems)}
              disabled={!selectedCategory || createMenuItemsMutation.isPending}
            >
              {createMenuItemsMutation.isPending ? (
                <>
                  <Loader2 className="mr-2 h-4 w-4 animate-spin" />
                  Ekleniyor...
                </>
              ) : (
                <>
                  <Save className="mr-2 h-4 w-4" />
                  Menü Öğelerini Ekle
                </>
              )}
            </Button>
          </>
        ) : (
          <Button 
            onClick={parseMenuItems}
            disabled={isParsing || !inputText.trim()}
          >
            {isParsing ? (
              <>
                <Loader2 className="mr-2 h-4 w-4 animate-spin" />
                Ayrıştırılıyor...
              </>
            ) : (
              <>
                <Upload className="mr-2 h-4 w-4" />
                İçeriği Ayrıştır
              </>
            )}
          </Button>
        )}
      </CardFooter>
    </Card>
  );
}