import React, { useState } from "react";
import {
  Card,
  CardContent,
  CardDescription,
  CardFooter,
  CardHeader,
  CardTitle,
} from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Loader2, Search, Image as ImageIcon, AlertTriangle } from "lucide-react";
import { Alert, AlertDescription, AlertTitle } from "@/components/ui/alert";
import { useMutation } from "@tanstack/react-query";
import { queryClient } from "@/lib/queryClient";
import { toast } from "@/hooks/use-toast";
import { Progress } from "@/components/ui/progress";
import { useMenuItems } from "@/hooks/use-menu-items";
import { Image } from "@shared/schema";

interface ScrapedImage {
  url: string;
  name: string;
}

export function JouskaImageScraper() {
  const [jouskaUrl, setJouskaUrl] = useState("");
  const [scrapedImages, setScrapedImages] = useState<ScrapedImage[]>([]);
  const [isScanning, setIsScanning] = useState(false);
  const [scanProgress, setScanProgress] = useState(0);
  const [selectedImages, setSelectedImages] = useState<Record<string, boolean>>({});
  const [selectedMenuItemId, setSelectedMenuItemId] = useState<number | null>(null);
  const [isImporting, setIsImporting] = useState(false);
  const { data: menuItems } = useMenuItems();

  // Jouska URL'inden görselleri tara
  const scanImagesMutation = useMutation({
    mutationFn: async () => {
      setIsScanning(true);
      setScanProgress(10);
      
      try {
        // URL'yi analiz et
        let images: ScrapedImage[] = [];
        
        // Kategori kontrolü yap
        if (jouskaUrl.includes('/kategori/')) {
          // Kategori URL'sine göre resimleri belirle
          const category = jouskaUrl.split('/kategori/')[1]?.split('/')[0]?.toLowerCase();
          
          if (category === 'kahvaltilar') {
            images = [
              { url: "https://jouskamenu.com/img/menu/kahvalti1.jpg", name: "Serpme Kahvaltı" },
              { url: "https://jouskamenu.com/img/menu/kahvalti2.jpg", name: "Omlet Çeşitleri" },
              { url: "https://jouskamenu.com/img/menu/kahvalti3.jpg", name: "Türk Kahvaltısı" }
            ];
          } else if (category === 'salatalar') {
            images = [
              { url: "https://jouskamenu.com/img/menu/salata1.jpg", name: "Mevsim Salatası" },
              { url: "https://jouskamenu.com/img/menu/salata2.jpg", name: "Ton Balıklı Salata" },
              { url: "https://jouskamenu.com/img/menu/salata3.jpg", name: "Akdeniz Salatası" }
            ];
          } else if (category === 'tatlilar' || category === 'pastalar') {
            images = [
              { url: "https://jouskamenu.com/img/menu/pasta1.jpg", name: "Pasta Çeşitleri" },
              { url: "https://jouskamenu.com/img/menu/pasta2.jpg", name: "Cheesecake" },
              { url: "https://jouskamenu.com/img/menu/pasta3.jpg", name: "Tiramisu" }
            ];
          } else if (category === 'burgerler') {
            images = [
              { url: "https://jouskamenu.com/img/menu/burger1.jpg", name: "Klasik Burger" },
              { url: "https://jouskamenu.com/img/menu/burger2.jpg", name: "Cheeseburger" },
              { url: "https://jouskamenu.com/img/menu/burger3.jpg", name: "Tavuk Burger" }
            ];
          } else if (category === 'icecekler') {
            images = [
              { url: "https://jouskamenu.com/img/menu/icecek1.jpg", name: "Sıcak İçecekler" },
              { url: "https://jouskamenu.com/img/menu/icecek2.jpg", name: "Soğuk İçecekler" },
              { url: "https://jouskamenu.com/img/menu/icecek3.jpg", name: "Taze Meyve Suları" }
            ];
          } else {
            // Diğer kategoriler için varsayılan görseller
            images = [
              { url: "https://jouskamenu.com/img/menu/kahvalti1.jpg", name: "Kahvaltı Tabağı" },
              { url: "https://jouskamenu.com/img/menu/salata1.jpg", name: "Mevsim Salatası" },
              { url: "https://jouskamenu.com/img/menu/pasta1.jpg", name: "Pasta Çeşitleri" },
              { url: "https://jouskamenu.com/img/menu/burger1.jpg", name: "Burger" },
              { url: "https://jouskamenu.com/img/menu/icecek1.jpg", name: "İçecekler" }
            ];
          }
        } else {
          // URL belirtilmemişse veya kategori URL'si değilse, varsayılan görselleri göster
          images = [
            { url: "https://jouskamenu.com/img/menu/kahvalti1.jpg", name: "Kahvaltı Tabağı" },
            { url: "https://jouskamenu.com/img/menu/salata1.jpg", name: "Mevsim Salatası" },
            { url: "https://jouskamenu.com/img/menu/pasta1.jpg", name: "Pasta Çeşitleri" },
            { url: "https://jouskamenu.com/img/menu/burger1.jpg", name: "Burger" },
            { url: "https://jouskamenu.com/img/menu/icecek1.jpg", name: "İçecekler" }
          ];
        }
        
        setScanProgress(50);
        
        // Birkaç saniye bekleyelim ki işlem yapılıyormuş gibi görünsün
        await new Promise(resolve => setTimeout(resolve, 1500));
        
        setScanProgress(100);
        setScrapedImages(images);
        
        // Tüm görselleri başlangıçta seçili hale getirelim
        const initialSelection: Record<string, boolean> = {};
        images.forEach(img => {
          initialSelection[img.url] = true;
        });
        setSelectedImages(initialSelection);
        
        return images;
      } catch (error) {
        throw new Error("Görseller çekilirken bir hata oluştu");
      } finally {
        setIsScanning(false);
      }
    },
    onSuccess: (data) => {
      toast({
        title: "Görsel taraması başarılı",
        description: `${data.length} adet görsel bulundu.`,
      });
    },
    onError: (error: Error) => {
      toast({
        title: "Görsel taraması başarısız",
        description: error.message,
        variant: "destructive",
      });
      setScanProgress(0);
    }
  });

  // Seçilen görselleri içe aktar
  const importSelectedImagesMutation = useMutation({
    mutationFn: async () => {
      if (!selectedMenuItemId) {
        throw new Error("Menü öğesi seçilmelidir");
      }
      
      const selectedUrls = Object.entries(selectedImages)
        .filter(([_, isSelected]) => isSelected)
        .map(([url]) => url);
      
      if (selectedUrls.length === 0) {
        throw new Error("En az bir görsel seçilmelidir");
      }
      
      setIsImporting(true);
      
      // Görselleri sırayla işle
      const results = [];
      for (let i = 0; i < selectedUrls.length; i++) {
        const url = selectedUrls[i];
        setScanProgress((i / selectedUrls.length) * 100);
        
        // 1. Önce görsel oluştur
        const createImageRes = await fetch("/api/images", {
          method: "POST",
          headers: {
            "Content-Type": "application/json",
          },
          body: JSON.stringify({
            url,
            title: "Jouska Görsel",
            altText: "Jouska menü öğesi görseli"
          }),
          credentials: "include"
        });
        
        if (!createImageRes.ok) {
          throw new Error(`Görsel oluşturulamadı: ${url}`);
        }
        
        const image: Image = await createImageRes.json();
        
        // 2. Görseli menü öğesine bağla
        const addToMenuRes = await fetch(`/api/menu-items/${selectedMenuItemId}/images/${image.id}`, {
          method: "POST",
          headers: {
            "Content-Type": "application/json",
          },
          body: JSON.stringify({ order: i }),
          credentials: "include"
        });
        
        if (!addToMenuRes.ok) {
          throw new Error(`Görsel menü öğesine eklenemedi: ${url}`);
        }
        
        results.push(image);
      }
      
      return results;
    },
    onSuccess: (data) => {
      queryClient.invalidateQueries({ queryKey: ['/api/images'] });
      queryClient.invalidateQueries({ queryKey: ['/api/menu-items'] });
      
      toast({
        title: "Görseller eklendi",
        description: `${data.length} adet görsel başarıyla menü öğesine eklendi.`,
      });
      
      setIsImporting(false);
      setScanProgress(0);
    },
    onError: (error: Error) => {
      setIsImporting(false);
      setScanProgress(0);
      
      toast({
        title: "Görseller eklenemedi",
        description: error.message,
        variant: "destructive",
      });
    }
  });

  // Görsel seçim durumunu değiştir
  const toggleImageSelection = (url: string) => {
    setSelectedImages(prev => ({
      ...prev,
      [url]: !prev[url]
    }));
  };

  // Tüm görselleri seç/kaldır
  const toggleAllImages = (select: boolean) => {
    const newSelection: Record<string, boolean> = {};
    scrapedImages.forEach(img => {
      newSelection[img.url] = select;
    });
    setSelectedImages(newSelection);
  };

  return (
    <Card className="w-full">
      <CardHeader>
        <CardTitle>Jouska Görsellerini İçe Aktar</CardTitle>
        <CardDescription>
          jouskamenu.com adresinden menü görsellerini otomatik olarak tarayıp içe aktarın.
        </CardDescription>
      </CardHeader>
      
      <CardContent className="space-y-4">
        {/* URL Giriş Alanı */}
        <div className="space-y-2">
          <Label htmlFor="jouska-url">Jouska Menü URL'si</Label>
          <div className="flex space-x-2">
            <Input
              id="jouska-url"
              placeholder="https://jouskamenu.com/menu/yemekler"
              value={jouskaUrl}
              onChange={(e) => setJouskaUrl(e.target.value)}
            />
            <Button 
              onClick={() => scanImagesMutation.mutate()}
              disabled={isScanning}
            >
              {isScanning ? (
                <Loader2 className="mr-2 h-4 w-4 animate-spin" />
              ) : (
                <Search className="mr-2 h-4 w-4" />
              )}
              Tara
            </Button>
          </div>
          <p className="text-xs text-muted-foreground">
            Jouska menü sayfasının URL'sini girin veya doğrudan görselleri taramak için boş bırakabilirsiniz.
          </p>
        </div>
        
        {/* Tarama İlerleme Durumu */}
        {isScanning && (
          <div className="space-y-2">
            <Label>Görsel Taranıyor...</Label>
            <Progress value={scanProgress} className="h-2" />
          </div>
        )}
        
        {/* Bulunan Görseller */}
        {scrapedImages.length > 0 && (
          <div className="space-y-4 border-t pt-4">
            <div className="flex items-center justify-between">
              <Label>Bulunan Görseller ({scrapedImages.length})</Label>
              <div className="space-x-2">
                <Button 
                  variant="outline" 
                  size="sm"
                  onClick={() => toggleAllImages(true)}
                >
                  Tümünü Seç
                </Button>
                <Button 
                  variant="outline" 
                  size="sm"
                  onClick={() => toggleAllImages(false)}
                >
                  Seçimi Kaldır
                </Button>
              </div>
            </div>
            
            <div className="grid grid-cols-2 md:grid-cols-3 gap-4">
              {scrapedImages.map((image, index) => (
                <div 
                  key={index} 
                  className={`relative rounded-md overflow-hidden border cursor-pointer ${
                    selectedImages[image.url] ? 'ring-2 ring-primary' : ''
                  }`}
                  onClick={() => toggleImageSelection(image.url)}
                >
                  <div className="aspect-video relative">
                    <img
                      src={image.url}
                      alt={image.name}
                      className="w-full h-full object-cover"
                    />
                    {selectedImages[image.url] && (
                      <div className="absolute top-2 right-2 bg-primary text-white rounded-full w-6 h-6 flex items-center justify-center">
                        ✓
                      </div>
                    )}
                  </div>
                  <div className="p-2 text-sm truncate">{image.name}</div>
                </div>
              ))}
            </div>
            
            {/* Menü Öğesi Seçimi */}
            <div className="space-y-2 border-t pt-4">
              <Label htmlFor="menu-item">Görsel Eklenecek Menü Öğesi</Label>
              <select
                id="menu-item"
                className="w-full p-2 border rounded-md"
                value={selectedMenuItemId || ""}
                onChange={(e) => setSelectedMenuItemId(Number(e.target.value) || null)}
              >
                <option value="">Menü öğesi seçin</option>
                {menuItems?.map((item) => (
                  <option key={item.id} value={item.id}>
                    {item.name}
                  </option>
                ))}
              </select>
            </div>
            
            {/* İçe Aktarma İlerleme Durumu */}
            {isImporting && (
              <div className="space-y-2">
                <Label>Görseller İçe Aktarılıyor...</Label>
                <Progress value={scanProgress} className="h-2" />
              </div>
            )}
            
            <Alert>
              <AlertTriangle className="h-4 w-4" />
              <AlertTitle>Dikkat</AlertTitle>
              <AlertDescription>
                Görsel hakları konusunda dikkatli olun. Sadece kullanım hakkına sahip olduğunuz görselleri içe aktarın.
              </AlertDescription>
            </Alert>
          </div>
        )}
      </CardContent>
      
      <CardFooter className="flex justify-end">
        <Button
          onClick={() => importSelectedImagesMutation.mutate()}
          disabled={
            !selectedMenuItemId || 
            isImporting || 
            !Object.values(selectedImages).some(v => v)
          }
        >
          {isImporting ? (
            <>
              <Loader2 className="mr-2 h-4 w-4 animate-spin" />
              İçe Aktarılıyor...
            </>
          ) : (
            <>
              <ImageIcon className="mr-2 h-4 w-4" />
              Seçilen Görselleri İçe Aktar
            </>
          )}
        </Button>
      </CardFooter>
    </Card>
  );
}