import React, { useState } from "react";
import {
  Card,
  CardContent,
  CardDescription,
  CardFooter,
  CardHeader,
  CardTitle,
} from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Textarea } from "@/components/ui/textarea";
import { 
  Alert, 
  AlertDescription, 
  AlertTitle 
} from "@/components/ui/alert";
import { Loader2, Save, Upload, AlertTriangle } from "lucide-react";
import { useMutation } from "@tanstack/react-query";
import { queryClient } from "@/lib/queryClient";
import { toast } from "@/hooks/use-toast";
import { useCategories } from "@/hooks/use-categories";

export function JouskaCategoryImporter() {
  const [isParsing, setIsParsing] = useState(false);
  const { data: categories, isLoading: categoriesLoading } = useCategories();

  // Jouska menüsündeki kategoriler (29 kategori)
  const jouskaCategories = [
    { name: "Kahvaltılar", slug: "kahvaltilar", order: 1 },
    { name: "Gözlemeler", slug: "gozlemeler", order: 2 },
    { name: "Omletler", slug: "omletler", order: 3 },
    { name: "Tostlar", slug: "tostlar", order: 4 },
    { name: "Burgerler", slug: "burgerler", order: 5 },
    { name: "Sandviçler", slug: "sandvicler", order: 6 },
    { name: "Makarnalar", slug: "makarnalar", order: 7 },
    { name: "Salatalar", slug: "salatalar", order: 8 },
    { name: "Ana Yemekler", slug: "ana-yemekler", order: 9 },
    { name: "Pizzalar", slug: "pizzalar", order: 10 },
    { name: "Deniz Ürünleri", slug: "deniz-urunleri", order: 11 },
    { name: "Et Yemekleri", slug: "et-yemekleri", order: 12 },
    { name: "Tavuk Yemekleri", slug: "tavuk-yemekleri", order: 13 },
    { name: "Çorbalar", slug: "corbalar", order: 14 },
    { name: "Mezeler", slug: "mezeler", order: 15 },
    { name: "Aperatifler", slug: "aperatifler", order: 16 },
    { name: "Tatlılar", slug: "tatlilar", order: 17 },
    { name: "Pasta ve Tartlar", slug: "pasta-ve-tartlar", order: 18 },
    { name: "Cheesecake", slug: "cheesecake", order: 19 },
    { name: "Waffle", slug: "waffle", order: 20 },
    { name: "Krep", slug: "krep", order: 21 },
    { name: "Dondurma", slug: "dondurma", order: 22 },
    { name: "Sütlü Tatlılar", slug: "sutlu-tatlilar", order: 23 },
    { name: "Sıcak İçecekler", slug: "sicak-icecekler", order: 24 },
    { name: "Türk Kahveleri", slug: "turk-kahveleri", order: 25 },
    { name: "Soğuk İçecekler", slug: "soguk-icecekler", order: 26 },
    { name: "Frappe ve Frozen", slug: "frappe-ve-frozen", order: 27 },
    { name: "Smoothie ve Shakes", slug: "smoothie-ve-shakes", order: 28 },
    { name: "Atıştırmalıklar", slug: "atistirmaliklar", order: 29 },
  ];

  // Import categories mutation
  const importCategoriesMutation = useMutation({
    mutationFn: async () => {
      setIsParsing(true);
      
      try {
        // Mevcut kategorileri kontrol et
        const existingCategories = categories || [];
        const existingSlugs = new Set(existingCategories.map(cat => cat.slug));
        
        // Sadece veritabanında olmayan kategorileri ekle
        const categoriesToAdd = jouskaCategories.filter(cat => !existingSlugs.has(cat.slug));
        
        if (categoriesToAdd.length === 0) {
          return { added: 0, message: "Tüm kategoriler zaten mevcut" };
        }
        
        // Kategorileri ekle
        for (const category of categoriesToAdd) {
          await fetch('/api/categories', {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify(category),
            credentials: 'include'
          });
        }
        
        return { 
          added: categoriesToAdd.length, 
          message: `${categoriesToAdd.length} kategori başarıyla eklendi` 
        };
      } catch (error) {
        throw new Error("Kategoriler eklenirken bir hata oluştu");
      } finally {
        setIsParsing(false);
      }
    },
    onSuccess: (data) => {
      queryClient.invalidateQueries({ queryKey: ['/api/categories'] });
      toast({
        title: "Kategoriler Eklendi",
        description: data.message,
      });
    },
    onError: (error: Error) => {
      toast({
        title: "Hata",
        description: error.message,
        variant: "destructive",
      });
    }
  });

  // Delete all categories mutation
  const deleteAllCategoriesMutation = useMutation({
    mutationFn: async () => {
      try {
        const existingCategories = categories || [];
        
        if (existingCategories.length === 0) {
          return { deleted: 0, message: "Silinecek kategori bulunamadı" };
        }
        
        // Her kategoriyi sil
        for (const category of existingCategories) {
          await fetch(`/api/categories/${category.id}`, {
            method: 'DELETE',
            credentials: 'include'
          });
        }
        
        return { 
          deleted: existingCategories.length, 
          message: `${existingCategories.length} kategori başarıyla silindi` 
        };
      } catch (error) {
        throw new Error("Kategoriler silinirken bir hata oluştu");
      }
    },
    onSuccess: (data) => {
      queryClient.invalidateQueries({ queryKey: ['/api/categories'] });
      toast({
        title: "Kategoriler Silindi",
        description: data.message,
      });
    },
    onError: (error: Error) => {
      toast({
        title: "Hata",
        description: error.message,
        variant: "destructive",
      });
    }
  });

  return (
    <Card className="w-full">
      <CardHeader>
        <CardTitle>Jouska Kategorileri İçe Aktar</CardTitle>
        <CardDescription>
          jouskamenu.com adresindeki tüm kategorileri tek tıklamayla içe aktarın.
        </CardDescription>
      </CardHeader>
      
      <CardContent>
        <div className="space-y-4">
          <Alert>
            <AlertTriangle className="h-4 w-4" />
            <AlertTitle>Bilgi</AlertTitle>
            <AlertDescription>
              Bu araç jouskamenu.com adresindeki kategori yapısını sisteminize aktaracaktır. 
              Aşağıdaki kategoriler içe aktarılacak:
            </AlertDescription>
          </Alert>
          
          <div className="grid grid-cols-2 gap-2 mt-4">
            {jouskaCategories.map((category, index) => (
              <div key={index} className="p-2 border rounded-md">
                <div className="font-medium">{category.name}</div>
                <div className="text-xs text-muted-foreground">Sıra: {category.order}</div>
              </div>
            ))}
          </div>
          
          <div className="mt-4">
            <h3 className="text-sm font-medium mb-2">Mevcut Kategoriler ({categories?.length || 0})</h3>
            {categoriesLoading ? (
              <div className="flex items-center justify-center p-4">
                <Loader2 className="animate-spin h-4 w-4 mr-2" />
                <span>Yükleniyor...</span>
              </div>
            ) : categories?.length ? (
              <div className="grid grid-cols-2 gap-2">
                {categories.map((category) => (
                  <div key={category.id} className="p-2 border rounded-md">
                    <div className="font-medium">{category.name}</div>
                    <div className="text-xs text-muted-foreground">Sıra: {category.order}</div>
                  </div>
                ))}
              </div>
            ) : (
              <div className="text-center p-4 text-muted-foreground">
                Henüz kategori bulunmuyor.
              </div>
            )}
          </div>
        </div>
      </CardContent>
      
      <CardFooter className="flex justify-between">
        <Button 
          variant="destructive" 
          onClick={() => {
            if (window.confirm("Tüm kategorileri silmek istediğinize emin misiniz?")) {
              deleteAllCategoriesMutation.mutate();
            }
          }}
          disabled={!categories?.length || deleteAllCategoriesMutation.isPending}
        >
          {deleteAllCategoriesMutation.isPending ? (
            <>
              <Loader2 className="mr-2 h-4 w-4 animate-spin" />
              Siliniyor...
            </>
          ) : (
            "Tüm Kategorileri Sil"
          )}
        </Button>
        
        <Button 
          onClick={() => importCategoriesMutation.mutate()}
          disabled={importCategoriesMutation.isPending}
        >
          {importCategoriesMutation.isPending ? (
            <>
              <Loader2 className="mr-2 h-4 w-4 animate-spin" />
              İçe Aktarılıyor...
            </>
          ) : (
            <>
              <Upload className="mr-2 h-4 w-4" />
              Kategorileri İçe Aktar
            </>
          )}
        </Button>
      </CardFooter>
    </Card>
  );
}