import React, { useState } from "react";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Textarea } from "@/components/ui/textarea";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { useToast } from "@/hooks/use-toast";
import { Dialog, DialogContent, DialogHeader, DialogTitle, DialogClose } from "@/components/ui/dialog";
import { TabsContent } from "@/components/ui/tabs";
import { Check, Copy, Edit, Plus, Trash2, X, Upload, ExternalLink, Image as ImageIcon } from "lucide-react";
import { ScrollArea } from "@/components/ui/scroll-area";
import { DropdownMenu, DropdownMenuContent, DropdownMenuItem, DropdownMenuTrigger } from "@/components/ui/dropdown-menu";
import { AlertDialog, AlertDialogAction, AlertDialogCancel, AlertDialogContent, AlertDialogDescription, AlertDialogFooter, AlertDialogHeader, AlertDialogTitle } from "@/components/ui/alert-dialog";

// Görsel tipi için interface
interface ImageItem {
  id: string;
  url: string;
  name: string;
  description?: string;
  size?: string;
  dateAdded: string;
}

export const ImageManager = () => {
  const { toast } = useToast();
  
  // Durum değişkenleri
  const [images, setImages] = useState<ImageItem[]>([]);
  const [selectedImage, setSelectedImage] = useState<ImageItem | null>(null);
  const [isAddDialogOpen, setIsAddDialogOpen] = useState(false);
  const [isDetailDialogOpen, setIsDetailDialogOpen] = useState(false);
  const [isDeleteDialogOpen, setIsDeleteDialogOpen] = useState(false);
  const [uploadProgress, setUploadProgress] = useState(0);
  const [isUploading, setIsUploading] = useState(false);
  const [newImageData, setNewImageData] = useState({
    name: "",
    description: "",
    url: ""
  });
  const [fileToUpload, setFileToUpload] = useState<File | null>(null);
  const [searchTerm, setSearchTerm] = useState("");
  const [showUploadSection, setShowUploadSection] = useState(false);
  
  // Yeni görsel ekle - URL ile
  const handleAddImage = () => {
    if (!newImageData.url || !newImageData.name) {
      toast({
        title: "Hata",
        description: "Görsel URL'si ve adı gereklidir",
        variant: "destructive"
      });
      return;
    }
    
    const newImage: ImageItem = {
      id: Date.now().toString(),
      url: newImageData.url,
      name: newImageData.name,
      description: newImageData.description,
      dateAdded: new Date().toISOString().slice(0, 10)
    };
    
    setImages([...images, newImage]);
    setNewImageData({ name: "", description: "", url: "" });
    setIsAddDialogOpen(false);
    
    toast({
      title: "Görsel Eklendi",
      description: `${newImage.name} başarıyla eklendi.`
    });
  };
  
  // Dosya yükleme işlemi - simülasyon
  const handleFileUpload = (e: React.ChangeEvent<HTMLInputElement>) => {
    const files = e.target.files;
    if (!files || files.length === 0) return;
    
    const file = files[0];
    setFileToUpload(file);
    setNewImageData({ 
      ...newImageData, 
      name: file.name.split('.')[0], 
    });
  };
  
  // Dosya yükleme işlemi
  const simulateUpload = () => {
    if (!fileToUpload) return;
    
    setIsUploading(true);
    setUploadProgress(0);
    
    try {
      // Dosya yükleme animasyonu
      const interval = setInterval(() => {
        setUploadProgress((prev) => {
          if (prev >= 100) {
            clearInterval(interval);
            return 100;
          }
          return prev + 10;
        });
      }, 100);
      
      // Dosya yükleme işlemi
      const reader = new FileReader();
      
      reader.onloadend = () => {
        // Yükleme tamamlandığında
        setTimeout(() => {
          clearInterval(interval);
          setIsUploading(false);
          
          try {
            // Base64 formatında görsel URL'si
            const imageUrl = reader.result as string;
            
            const newImage: ImageItem = {
              id: Date.now().toString(),
              url: imageUrl,
              name: newImageData.name || fileToUpload.name.split('.')[0],
              description: newImageData.description,
              size: `${(fileToUpload.size / 1024).toFixed(2)} KB`,
              dateAdded: new Date().toISOString().slice(0, 10)
            };
            
            setImages([...images, newImage]);
            setNewImageData({ name: "", description: "", url: "" });
            setFileToUpload(null);
            setShowUploadSection(false);
            
            toast({
              title: "Görsel Yüklendi",
              description: `${newImage.name} başarıyla yüklendi.`
            });
          } catch (error) {
            console.error("Görsel işleme hatası:", error);
            toast({
              title: "Görsel Yükleme Hatası",
              description: "Görsel yüklenirken bir hata oluştu.",
              variant: "destructive"
            });
          }
        }, 500);
      };
      
      reader.onerror = () => {
        clearInterval(interval);
        setIsUploading(false);
        toast({
          title: "Görsel Yükleme Hatası",
          description: "Görsel okuma sırasında bir hata oluştu.",
          variant: "destructive"
        });
      };
      
      // Dosyayı oku
      reader.readAsDataURL(fileToUpload);
      
    } catch (error) {
      console.error("Görsel yükleme hatası:", error);
      setIsUploading(false);
      setUploadProgress(0);
      toast({
        title: "Görsel Yükleme Hatası",
        description: "Görsel yüklenirken beklenmeyen bir hata oluştu.",
        variant: "destructive"
      });
    }
  };
  
  // Görsel detaylarını göster
  const handleViewDetails = (image: ImageItem) => {
    setSelectedImage(image);
    setIsDetailDialogOpen(true);
  };
  
  // Görsel silme işlemi
  const handleDeleteImage = () => {
    if (!selectedImage) return;
    
    const updatedImages = images.filter(img => img.id !== selectedImage.id);
    setImages(updatedImages);
    setIsDeleteDialogOpen(false);
    
    toast({
      title: "Görsel Silindi",
      description: `${selectedImage.name} başarıyla silindi.`
    });
  };
  
  // Görsel URL'sini panoya kopyala
  const handleCopyUrl = (url: string) => {
    navigator.clipboard.writeText(url).then(() => {
      toast({
        title: "URL Kopyalandı",
        description: "Görsel URL'si panoya kopyalandı."
      });
    });
  };
  
  // Arama sonucuna göre filtreleme
  const filteredImages = images.filter(image => 
    image.name.toLowerCase().includes(searchTerm.toLowerCase()) ||
    (image.description && image.description.toLowerCase().includes(searchTerm.toLowerCase()))
  );
  
  return (
    <div className="space-y-6">
      <div className="flex justify-between items-center">
        <h2 className="text-2xl font-bold text-primary font-display">Görsel Yönetimi</h2>
        <div className="flex space-x-2">
          <Button variant="outline" onClick={() => setShowUploadSection(!showUploadSection)}>
            <Upload className="h-4 w-4 mr-2" />
            Görsel Yükle
          </Button>
          <Button onClick={() => setIsAddDialogOpen(true)}>
            <Plus className="h-4 w-4 mr-2" />
            URL ile Ekle
          </Button>
        </div>
      </div>
      
      {/* Görsel yükleme bölümü */}
      {showUploadSection && (
        <Card className="border-dashed border-primary/50">
          <CardContent className="pt-6">
            <div className="text-center py-4">
              <div className="mx-auto w-32 h-32 rounded-lg mb-4 border-2 border-dashed border-gray-300 flex items-center justify-center bg-gray-50">
                {fileToUpload ? (
                  <div className="relative w-full h-full overflow-hidden">
                    <img 
                      src={URL.createObjectURL(fileToUpload)} 
                      alt="Preview" 
                      className="object-cover w-full h-full"
                    />
                    <button 
                      className="absolute top-1 right-1 bg-white rounded-full p-1"
                      onClick={() => setFileToUpload(null)}
                    >
                      <X className="h-3 w-3" />
                    </button>
                  </div>
                ) : (
                  <ImageIcon className="h-10 w-10 text-gray-400" />
                )}
              </div>
              
              <div className="space-y-3 max-w-md mx-auto">
                {!fileToUpload ? (
                  <div className="flex justify-center">
                    <label htmlFor="file-upload" className="cursor-pointer bg-primary/5 hover:bg-primary/10 px-4 py-2 rounded-md text-primary transition-colors">
                      Dosya Seç
                      <Input
                        id="file-upload"
                        type="file"
                        accept="image/*"
                        onChange={handleFileUpload}
                        className="hidden"
                      />
                    </label>
                  </div>
                ) : (
                  <div className="space-y-4">
                    <div className="space-y-2">
                      <Input
                        placeholder="Görsel adı"
                        value={newImageData.name}
                        onChange={(e) => setNewImageData({...newImageData, name: e.target.value})}
                      />
                      <Textarea
                        placeholder="Açıklama (isteğe bağlı)"
                        value={newImageData.description}
                        onChange={(e) => setNewImageData({...newImageData, description: e.target.value})}
                        rows={2}
                      />
                    </div>
                    
                    {isUploading && (
                      <div className="w-full bg-gray-200 rounded-full h-2.5">
                        <div className="bg-primary h-2.5 rounded-full" style={{ width: `${uploadProgress}%` }}></div>
                      </div>
                    )}
                    
                    <div className="flex space-x-2">
                      <Button 
                        variant="default" 
                        onClick={simulateUpload} 
                        disabled={isUploading}
                        className="flex-1"
                      >
                        {isUploading ? 'Yükleniyor...' : 'Yükle'}
                      </Button>
                      <Button 
                        variant="outline" 
                        onClick={() => setShowUploadSection(false)}
                        disabled={isUploading}
                      >
                        İptal
                      </Button>
                    </div>
                  </div>
                )}
              </div>
            </div>
          </CardContent>
        </Card>
      )}
      
      <Card>
        <CardHeader>
          <div className="flex justify-between items-center">
            <div>
              <CardTitle>Görseller</CardTitle>
              <CardDescription>
                Menü öğeleri için kullanabileceğiniz görsel koleksiyonu
              </CardDescription>
            </div>
            <div className="w-64">
              <Input 
                placeholder="Görsel ara..." 
                value={searchTerm}
                onChange={(e) => setSearchTerm(e.target.value)}
              />
            </div>
          </div>
        </CardHeader>
        <CardContent>
          {filteredImages.length === 0 ? (
            <div className="text-center py-10 text-gray-500">
              <ImageIcon className="mx-auto h-10 w-10 mb-2 text-gray-400" />
              {images.length === 0 ? (
                <p>Henüz görsel eklenmedi. Görsel eklemek için "Görsel Yükle" veya "URL ile Ekle" butonunu kullanın.</p>
              ) : (
                <p>Arama kriterlerine uygun görsel bulunamadı.</p>
              )}
            </div>
          ) : (
            <div className="grid grid-cols-2 sm:grid-cols-3 md:grid-cols-4 lg:grid-cols-5 gap-4">
              {filteredImages.map((image) => (
                <div key={image.id} className="group relative overflow-hidden rounded-md border border-gray-200 bg-white">
                  <div className="aspect-square overflow-hidden bg-gray-100 flex items-center justify-center">
                    <img 
                      src={image.url} 
                      alt={image.name} 
                      className="object-cover w-full h-full hover:scale-105 transition-transform duration-300"
                      onClick={() => handleViewDetails(image)}
                    />
                  </div>
                  <div className="p-2">
                    <h3 className="text-sm font-medium truncate" title={image.name}>{image.name}</h3>
                    <p className="text-xs text-gray-500">{image.dateAdded}</p>
                  </div>
                  <div className="absolute top-2 right-2 opacity-0 group-hover:opacity-100 transition-opacity">
                    <DropdownMenu>
                      <DropdownMenuTrigger asChild>
                        <Button variant="secondary" size="sm" className="h-8 w-8 p-0 rounded-full">
                          <Edit className="h-4 w-4" />
                        </Button>
                      </DropdownMenuTrigger>
                      <DropdownMenuContent>
                        <DropdownMenuItem onClick={() => handleViewDetails(image)}>
                          <div className="flex items-center">
                            <ImageIcon className="mr-2 h-4 w-4" />
                            <span>Detaylar</span>
                          </div>
                        </DropdownMenuItem>
                        <DropdownMenuItem onClick={() => handleCopyUrl(image.url)}>
                          <div className="flex items-center">
                            <Copy className="mr-2 h-4 w-4" />
                            <span>URL Kopyala</span>
                          </div>
                        </DropdownMenuItem>
                        <DropdownMenuItem 
                          onClick={() => {
                            setSelectedImage(image);
                            setIsDeleteDialogOpen(true);
                          }}
                          className="text-red-500 focus:text-red-500"
                        >
                          <div className="flex items-center">
                            <Trash2 className="mr-2 h-4 w-4" />
                            <span>Sil</span>
                          </div>
                        </DropdownMenuItem>
                      </DropdownMenuContent>
                    </DropdownMenu>
                  </div>
                </div>
              ))}
            </div>
          )}
        </CardContent>
      </Card>
      
      {/* URL ile görsel ekleme dialog */}
      <Dialog open={isAddDialogOpen} onOpenChange={setIsAddDialogOpen}>
        <DialogContent>
          <DialogHeader>
            <DialogTitle>URL ile Görsel Ekle</DialogTitle>
            <DialogClose className="absolute right-4 top-4">
              <X className="h-4 w-4" />
            </DialogClose>
          </DialogHeader>
          <div className="space-y-4 py-4">
            <div className="space-y-2">
              <label htmlFor="url">Görsel URL'si</label>
              <Input
                id="url"
                value={newImageData.url}
                onChange={(e) => setNewImageData({...newImageData, url: e.target.value})}
                placeholder="https://example.com/image.jpg"
              />
            </div>
            
            <div className="space-y-2">
              <label htmlFor="name">Görsel Adı</label>
              <Input
                id="name"
                value={newImageData.name}
                onChange={(e) => setNewImageData({...newImageData, name: e.target.value})}
                placeholder="Menü öğesi görseli"
              />
            </div>
            
            <div className="space-y-2">
              <label htmlFor="description">Açıklama (İsteğe Bağlı)</label>
              <Textarea
                id="description"
                value={newImageData.description}
                onChange={(e) => setNewImageData({...newImageData, description: e.target.value})}
                placeholder="Görsel hakkında kısa bir açıklama"
                rows={3}
              />
            </div>
            
            {newImageData.url && (
              <div className="mt-4 border rounded-md overflow-hidden">
                <img 
                  src={newImageData.url} 
                  alt="Preview" 
                  className="max-h-40 mx-auto object-contain"
                  onError={(e) => {
                    // Inline SVG kullanarak "Görsel yüklenemedi" görseli oluştur
                    const svgString = `
                      <svg xmlns="http://www.w3.org/2000/svg" width="200" height="120" viewBox="0 0 200 120">
                        <rect width="200" height="120" fill="#f0f0f0" />
                        <text x="100" y="60" font-family="Arial" font-size="12" text-anchor="middle">Görsel Yüklenemedi</text>
                      </svg>
                    `;
                    const encodedSvg = encodeURIComponent(svgString);
                    e.currentTarget.src = `data:image/svg+xml;charset=utf-8,${encodedSvg}`;
                    e.currentTarget.alt = 'Görsel yüklenemedi';
                  }}
                />
              </div>
            )}
          </div>
          <div className="flex justify-end space-x-2">
            <Button variant="outline" onClick={() => setIsAddDialogOpen(false)}>
              İptal
            </Button>
            <Button onClick={handleAddImage}>
              Görsel Ekle
            </Button>
          </div>
        </DialogContent>
      </Dialog>
      
      {/* Görsel detay dialog */}
      <Dialog open={isDetailDialogOpen} onOpenChange={setIsDetailDialogOpen}>
        <DialogContent className="max-w-3xl">
          <DialogHeader>
            <DialogTitle>Görsel Detayları</DialogTitle>
            <DialogClose className="absolute right-4 top-4">
              <X className="h-4 w-4" />
            </DialogClose>
          </DialogHeader>
          {selectedImage && (
            <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
              <div className="bg-gray-50 rounded-md overflow-hidden flex items-center justify-center p-2">
                <img 
                  src={selectedImage.url} 
                  alt={selectedImage.name} 
                  className="max-w-full max-h-[400px] object-contain"
                />
              </div>
              <div className="space-y-4">
                <div>
                  <h3 className="text-lg font-semibold">{selectedImage.name}</h3>
                  {selectedImage.description && (
                    <p className="text-gray-600 mt-1">{selectedImage.description}</p>
                  )}
                </div>
                
                <div className="pt-4 space-y-2 border-t border-gray-200">
                  <div className="flex justify-between text-sm">
                    <span className="text-gray-500">Eklenme Tarihi</span>
                    <span>{selectedImage.dateAdded}</span>
                  </div>
                  {selectedImage.size && (
                    <div className="flex justify-between text-sm">
                      <span className="text-gray-500">Boyut</span>
                      <span>{selectedImage.size}</span>
                    </div>
                  )}
                </div>
                
                <div className="pt-4 space-y-2 border-t border-gray-200">
                  <label className="text-sm text-gray-500">Görsel URL'si</label>
                  <div className="flex">
                    <Input 
                      value={selectedImage.url} 
                      readOnly 
                      className="rounded-r-none"
                    />
                    <Button 
                      variant="outline" 
                      className="rounded-l-none border-l-0"
                      onClick={() => handleCopyUrl(selectedImage.url)}
                    >
                      <Copy className="h-4 w-4" />
                    </Button>
                  </div>
                </div>
                
                <div className="pt-4 flex justify-between items-center">
                  <Button variant="outline" className="text-red-500" onClick={() => {
                    setIsDetailDialogOpen(false);
                    setIsDeleteDialogOpen(true);
                  }}>
                    <Trash2 className="h-4 w-4 mr-2" />
                    Görseli Sil
                  </Button>
                  
                  <Button onClick={() => window.open(selectedImage.url, "_blank")}>
                    <ExternalLink className="h-4 w-4 mr-2" />
                    Yeni Sekmede Aç
                  </Button>
                </div>
              </div>
            </div>
          )}
        </DialogContent>
      </Dialog>
      
      {/* Görsel silme dialog */}
      <AlertDialog open={isDeleteDialogOpen} onOpenChange={setIsDeleteDialogOpen}>
        <AlertDialogContent>
          <AlertDialogHeader>
            <AlertDialogTitle>Görseli Sil</AlertDialogTitle>
            <AlertDialogDescription>
              <p>
                <strong>{selectedImage?.name}</strong> görselini silmek istediğinizden emin misiniz?
              </p>
              <p className="mt-2 text-red-500">
                Bu işlem geri alınamaz ve görseli kullanan menü öğeleri artık bu görseli gösteremeyecektir.
              </p>
            </AlertDialogDescription>
          </AlertDialogHeader>
          <AlertDialogFooter>
            <AlertDialogCancel>İptal</AlertDialogCancel>
            <AlertDialogAction
              onClick={handleDeleteImage}
              className="bg-red-500 text-white hover:bg-red-600"
            >
              Sil
            </AlertDialogAction>
          </AlertDialogFooter>
        </AlertDialogContent>
      </AlertDialog>
    </div>
  );
};