import React, { useState } from "react";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Badge } from "@/components/ui/badge";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { Dialog, DialogContent, DialogDescription, DialogFooter, DialogHeader, DialogTitle, DialogTrigger } from "@/components/ui/dialog";
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow } from "@/components/ui/table";
import { Switch } from "@/components/ui/switch";
import { useToast } from "@/hooks/use-toast";
import { useMutation, useQuery, useQueryClient } from "@tanstack/react-query";
import { apiRequest } from "@/lib/queryClient";
import { Coupon, InsertCoupon } from "@shared/schema";
import { 
  Gift, 
  Plus, 
  Search, 
  Edit, 
  Trash2, 
  Calendar, 
  Percent,
  DollarSign,
  Users,
  Copy
} from "lucide-react";

export default function CouponManagement() {
  const { toast } = useToast();
  const queryClient = useQueryClient();
  const [searchTerm, setSearchTerm] = useState("");
  const [isAddDialogOpen, setIsAddDialogOpen] = useState(false);
  const [selectedCoupon, setSelectedCoupon] = useState<Coupon | null>(null);
  
  const [newCoupon, setNewCoupon] = useState<InsertCoupon>({
    code: "",
    name: "",
    discountType: "percentage",
    discountValue: "",
    minOrderAmount: "0",
    maxUses: 1,
    isActive: true,
  });

  // Fetch coupons
  const { data: coupons = [], isLoading } = useQuery<Coupon[]>({
    queryKey: ["/api/coupons"],
  });

  // Create coupon mutation
  const createCouponMutation = useMutation({
    mutationFn: async (coupon: InsertCoupon) => {
      const response = await apiRequest("POST", "/api/coupons", coupon);
      return response.json();
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/coupons"] });
      toast({
        title: "Başarılı",
        description: "Kupon başarıyla oluşturuldu",
      });
      setIsAddDialogOpen(false);
      resetNewCoupon();
    },
    onError: (error: Error) => {
      toast({
        title: "Hata",
        description: error.message,
        variant: "destructive",
      });
    },
  });

  // Update coupon mutation
  const updateCouponMutation = useMutation({
    mutationFn: async ({ id, ...coupon }: { id: number } & Partial<InsertCoupon>) => {
      const response = await apiRequest("PATCH", `/api/coupons/${id}`, coupon);
      return response.json();
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/coupons"] });
      toast({
        title: "Başarılı",
        description: "Kupon güncellendi",
      });
      setSelectedCoupon(null);
    },
    onError: (error: Error) => {
      toast({
        title: "Hata",
        description: error.message,
        variant: "destructive",
      });
    },
  });

  // Delete coupon mutation
  const deleteCouponMutation = useMutation({
    mutationFn: async (id: number) => {
      await apiRequest("DELETE", `/api/coupons/${id}`);
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/coupons"] });
      toast({
        title: "Başarılı",
        description: "Kupon silindi",
      });
    },
    onError: (error: Error) => {
      toast({
        title: "Hata",
        description: error.message,
        variant: "destructive",
      });
    },
  });

  const resetNewCoupon = () => {
    setNewCoupon({
      code: "",
      name: "",
      discountType: "percentage",
      discountValue: "",
      minOrderAmount: "0",
      maxUses: 1,
      isActive: true,
    });
  };

  const generateCouponCode = () => {
    const code = Math.random().toString(36).substring(2, 8).toUpperCase();
    setNewCoupon({ ...newCoupon, code });
  };

  const filteredCoupons = coupons.filter(coupon =>
    coupon.code.toLowerCase().includes(searchTerm.toLowerCase()) ||
    coupon.name.toLowerCase().includes(searchTerm.toLowerCase())
  );

  const handleCreateCoupon = () => {
    createCouponMutation.mutate(newCoupon);
  };

  const handleUpdateCoupon = () => {
    if (selectedCoupon) {
      updateCouponMutation.mutate({
        id: selectedCoupon.id,
        code: selectedCoupon.code,
        name: selectedCoupon.name,
        discountType: selectedCoupon.discountType,
        discountValue: selectedCoupon.discountValue,
        minOrderAmount: selectedCoupon.minOrderAmount,
        maxUses: selectedCoupon.maxUses,
        isActive: selectedCoupon.isActive,
      });
    }
  };

  const handleDeleteCoupon = (id: number) => {
    if (confirm("Bu kuponu silmek istediğinizden emin misiniz?")) {
      deleteCouponMutation.mutate(id);
    }
  };

  const copyToClipboard = (text: string) => {
    navigator.clipboard.writeText(text);
    toast({
      title: "Kopyalandı",
      description: "Kupon kodu panoya kopyalandı",
    });
  };

  return (
    <div className="space-y-6">
      {/* Header */}
      <div className="flex justify-between items-center">
        <div>
          <h2 className="text-2xl font-bold tracking-tight">Kupon Yönetimi</h2>
          <p className="text-muted-foreground">
            İndirim kuponları oluşturun ve yönetin
          </p>
        </div>
        <Dialog open={isAddDialogOpen} onOpenChange={setIsAddDialogOpen}>
          <DialogTrigger asChild>
            <Button>
              <Plus className="h-4 w-4 mr-2" />
              Yeni Kupon
            </Button>
          </DialogTrigger>
          <DialogContent className="max-w-md">
            <DialogHeader>
              <DialogTitle>Yeni Kupon Oluştur</DialogTitle>
              <DialogDescription>
                İndirim kuponu bilgilerini girin
              </DialogDescription>
            </DialogHeader>
            <div className="grid gap-4 py-4">
              <div className="space-y-2">
                <Label htmlFor="code">Kupon Kodu</Label>
                <div className="flex space-x-2">
                  <Input
                    id="code"
                    value={newCoupon.code}
                    onChange={(e) => setNewCoupon({ ...newCoupon, code: e.target.value.toUpperCase() })}
                    placeholder="KUPON123"
                  />
                  <Button type="button" variant="outline" onClick={generateCouponCode}>
                    Üret
                  </Button>
                </div>
              </div>
              <div className="space-y-2">
                <Label htmlFor="name">Kupon Adı</Label>
                <Input
                  id="name"
                  value={newCoupon.name}
                  onChange={(e) => setNewCoupon({ ...newCoupon, name: e.target.value })}
                  placeholder="Yılbaşı İndirimi"
                />
              </div>
              <div className="space-y-2">
                <Label htmlFor="discount-type">İndirim Türü</Label>
                <Select value={newCoupon.discountType} onValueChange={(value: "percentage" | "fixed") => setNewCoupon({ ...newCoupon, discountType: value })}>
                  <SelectTrigger>
                    <SelectValue />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="percentage">Yüzde (%)</SelectItem>
                    <SelectItem value="fixed">Sabit Tutar (TL)</SelectItem>
                  </SelectContent>
                </Select>
              </div>
              <div className="space-y-2">
                <Label htmlFor="discount-value">İndirim Miktarı</Label>
                <Input
                  id="discount-value"
                  type="number"
                  value={newCoupon.discountValue}
                  onChange={(e) => setNewCoupon({ ...newCoupon, discountValue: e.target.value })}
                  placeholder={newCoupon.discountType === "percentage" ? "10" : "25"}
                />
              </div>
              <div className="space-y-2">
                <Label htmlFor="min-order">Minimum Sipariş Tutarı (TL)</Label>
                <Input
                  id="min-order"
                  type="number"
                  value={newCoupon.minOrderAmount || ""}
                  onChange={(e) => setNewCoupon({ ...newCoupon, minOrderAmount: e.target.value })}
                  placeholder="0"
                />
              </div>
              <div className="space-y-2">
                <Label htmlFor="max-uses">Maksimum Kullanım</Label>
                <Input
                  id="max-uses"
                  type="number"
                  value={newCoupon.maxUses || ""}
                  onChange={(e) => setNewCoupon({ ...newCoupon, maxUses: parseInt(e.target.value) || 1 })}
                  placeholder="1"
                />
              </div>
              <div className="flex items-center space-x-2">
                <Switch
                  id="is-active"
                  checked={newCoupon.isActive || false}
                  onCheckedChange={(checked) => setNewCoupon({ ...newCoupon, isActive: checked })}
                />
                <Label htmlFor="is-active">Aktif</Label>
              </div>
            </div>
            <DialogFooter>
              <Button onClick={handleCreateCoupon} disabled={createCouponMutation.isPending}>
                {createCouponMutation.isPending ? "Oluşturuluyor..." : "Kupon Oluştur"}
              </Button>
            </DialogFooter>
          </DialogContent>
        </Dialog>
      </div>

      {/* Search and Stats */}
      <div className="grid grid-cols-1 md:grid-cols-4 gap-4">
        <Card className="md:col-span-2">
          <CardContent className="p-4">
            <div className="relative">
              <Search className="absolute left-3 top-1/2 transform -translate-y-1/2 text-gray-400 h-4 w-4" />
              <Input
                placeholder="Kupon ara..."
                value={searchTerm}
                onChange={(e) => setSearchTerm(e.target.value)}
                className="pl-10"
              />
            </div>
          </CardContent>
        </Card>
        
        <Card>
          <CardContent className="p-4">
            <div className="flex items-center space-x-2">
              <Gift className="h-5 w-5 text-blue-600" />
              <div>
                <p className="text-sm text-muted-foreground">Toplam Kupon</p>
                <p className="text-2xl font-bold">{coupons.length}</p>
              </div>
            </div>
          </CardContent>
        </Card>
        
        <Card>
          <CardContent className="p-4">
            <div className="flex items-center space-x-2">
              <Users className="h-5 w-5 text-green-600" />
              <div>
                <p className="text-sm text-muted-foreground">Aktif Kupon</p>
                <p className="text-2xl font-bold">{coupons.filter(c => c.isActive).length}</p>
              </div>
            </div>
          </CardContent>
        </Card>
      </div>

      {/* Coupon Table */}
      <Card>
        <CardHeader>
          <CardTitle className="flex items-center">
            <Gift className="h-5 w-5 mr-2" />
            Kupon Listesi
          </CardTitle>
          <CardDescription>
            {filteredCoupons.length} kupon görüntüleniyor
          </CardDescription>
        </CardHeader>
        <CardContent>
          <Table>
            <TableHeader>
              <TableRow>
                <TableHead>Kupon</TableHead>
                <TableHead>İndirim</TableHead>
                <TableHead>Koşullar</TableHead>
                <TableHead>Kullanım</TableHead>
                <TableHead>Durum</TableHead>
                <TableHead>İşlemler</TableHead>
              </TableRow>
            </TableHeader>
            <TableBody>
              {filteredCoupons.map((coupon) => (
                <TableRow key={coupon.id}>
                  <TableCell>
                    <div>
                      <div className="flex items-center space-x-2">
                        <p className="font-mono font-bold">{coupon.code}</p>
                        <Button
                          variant="ghost"
                          size="sm"
                          onClick={() => copyToClipboard(coupon.code)}
                        >
                          <Copy className="h-3 w-3" />
                        </Button>
                      </div>
                      <p className="text-sm text-muted-foreground">{coupon.name}</p>
                    </div>
                  </TableCell>
                  <TableCell>
                    <div className="flex items-center">
                      {coupon.discountType === "percentage" ? (
                        <>
                          <Percent className="h-4 w-4 mr-1" />
                          {coupon.discountValue}%
                        </>
                      ) : (
                        <>
                          <DollarSign className="h-4 w-4 mr-1" />
                          {coupon.discountValue} TL
                        </>
                      )}
                    </div>
                  </TableCell>
                  <TableCell>
                    <div className="text-sm">
                      <p>Min: {coupon.minOrderAmount} TL</p>
                      <p>Max kullanım: {coupon.maxUses}</p>
                    </div>
                  </TableCell>
                  <TableCell>
                    <Badge variant={Number(coupon.usedCount) > 0 ? "default" : "secondary"}>
                      {coupon.usedCount || 0} / {coupon.maxUses}
                    </Badge>
                  </TableCell>
                  <TableCell>
                    <Badge variant={coupon.isActive ? "default" : "secondary"}>
                      {coupon.isActive ? "Aktif" : "Pasif"}
                    </Badge>
                  </TableCell>
                  <TableCell>
                    <div className="flex space-x-2">
                      <Button
                        variant="outline"
                        size="sm"
                        onClick={() => setSelectedCoupon(coupon)}
                      >
                        <Edit className="h-4 w-4" />
                      </Button>
                      <Button
                        variant="outline"
                        size="sm"
                        onClick={() => handleDeleteCoupon(coupon.id)}
                        disabled={deleteCouponMutation.isPending}
                      >
                        <Trash2 className="h-4 w-4" />
                      </Button>
                    </div>
                  </TableCell>
                </TableRow>
              ))}
            </TableBody>
          </Table>
        </CardContent>
      </Card>

      {/* Edit Coupon Dialog */}
      <Dialog open={!!selectedCoupon} onOpenChange={() => setSelectedCoupon(null)}>
        <DialogContent className="max-w-md">
          <DialogHeader>
            <DialogTitle>Kupon Düzenle</DialogTitle>
            <DialogDescription>
              Kupon bilgilerini güncelleyin
            </DialogDescription>
          </DialogHeader>
          {selectedCoupon && (
            <div className="grid gap-4 py-4">
              <div className="space-y-2">
                <Label htmlFor="edit-code">Kupon Kodu</Label>
                <Input
                  id="edit-code"
                  value={selectedCoupon.code}
                  onChange={(e) => setSelectedCoupon({ ...selectedCoupon, code: e.target.value.toUpperCase() })}
                />
              </div>
              <div className="space-y-2">
                <Label htmlFor="edit-name">Kupon Adı</Label>
                <Input
                  id="edit-name"
                  value={selectedCoupon.name}
                  onChange={(e) => setSelectedCoupon({ ...selectedCoupon, name: e.target.value })}
                />
              </div>
              <div className="space-y-2">
                <Label htmlFor="edit-discount-type">İndirim Türü</Label>
                <Select value={selectedCoupon.discountType} onValueChange={(value: "percentage" | "fixed") => setSelectedCoupon({ ...selectedCoupon, discountType: value })}>
                  <SelectTrigger>
                    <SelectValue />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="percentage">Yüzde (%)</SelectItem>
                    <SelectItem value="fixed">Sabit Tutar (TL)</SelectItem>
                  </SelectContent>
                </Select>
              </div>
              <div className="space-y-2">
                <Label htmlFor="edit-discount-value">İndirim Miktarı</Label>
                <Input
                  id="edit-discount-value"
                  type="number"
                  value={selectedCoupon.discountValue}
                  onChange={(e) => setSelectedCoupon({ ...selectedCoupon, discountValue: e.target.value })}
                />
              </div>
              <div className="space-y-2">
                <Label htmlFor="edit-min-order">Minimum Sipariş Tutarı (TL)</Label>
                <Input
                  id="edit-min-order"
                  type="number"
                  value={selectedCoupon.minOrderAmount || ""}
                  onChange={(e) => setSelectedCoupon({ ...selectedCoupon, minOrderAmount: e.target.value })}
                />
              </div>
              <div className="space-y-2">
                <Label htmlFor="edit-max-uses">Maksimum Kullanım</Label>
                <Input
                  id="edit-max-uses"
                  type="number"
                  value={String(selectedCoupon.maxUses || "")}
                  onChange={(e) => setSelectedCoupon({ ...selectedCoupon, maxUses: parseInt(e.target.value) || 1 })}
                />
              </div>
              <div className="flex items-center space-x-2">
                <Switch
                  id="edit-is-active"
                  checked={selectedCoupon.isActive || false}
                  onCheckedChange={(checked) => setSelectedCoupon({ ...selectedCoupon, isActive: checked })}
                />
                <Label htmlFor="edit-is-active">Aktif</Label>
              </div>
            </div>
          )}
          <DialogFooter>
            <Button onClick={handleUpdateCoupon} disabled={updateCouponMutation.isPending}>
              {updateCouponMutation.isPending ? "Güncelleniyor..." : "Güncelle"}
            </Button>
          </DialogFooter>
        </DialogContent>
      </Dialog>
    </div>
  );
}