import React, { useState } from "react";
import { Card, CardContent, CardDescription, CardFooter, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Textarea } from "@/components/ui/textarea";
import { Separator } from "@/components/ui/separator";
import { useToast } from "@/hooks/use-toast";
import { Loader2, Save, Plus, X } from "lucide-react";
import { useCategories } from "@/hooks/use-categories";
import { useMutation } from "@tanstack/react-query";
import { queryClient, apiRequest } from "@/lib/queryClient";
import { Category } from "@shared/schema";

// Jouska kategorileri
const jouskaCategoryData = [
  { name: "Kahvaltılar", slug: "kahvaltilar", order: 1 },
  { name: "Omletler", slug: "omletler", order: 2 },
  { name: "Gözlemeler", slug: "gozlemeler", order: 3 },
  { name: "Makarnalar", slug: "makarnalar", order: 4 },
  { name: "Salatalar", slug: "salatalar", order: 5 },
  { name: "Tostlar", slug: "tostlar", order: 6 },
  { name: "Burgerler", slug: "burgerler", order: 7 },
  { name: "Waffle", slug: "waffle", order: 8 },
  { name: "Kremalar", slug: "kremalar", order: 9 },
  { name: "Cheesecake", slug: "cheesecake", order: 10 },
  { name: "Pasta ve Tartlar", slug: "pasta-ve-tartlar", order: 11 },
  { name: "Sütlü Tatlılar", slug: "sutlu-tatlilar", order: 12 },
  { name: "Dondurma", slug: "dondurma", order: 13 },
  { name: "Soğuk İçecekler", slug: "soguk-icecekler", order: 14 },
  { name: "Sıcak İçecekler", slug: "sicak-icecekler", order: 15 },
  { name: "Atıştırmalıklar", slug: "atistirmaliklar", order: 16 },
];

interface CategoryEditorProps {
  category: {
    name: string;
    slug: string;
    order: number;
  };
  onDelete: () => void;
  onChange: (updatedCategory: {name: string; slug: string; order: number}) => void;
}

const CategoryEditor: React.FC<CategoryEditorProps> = ({ category, onDelete, onChange }) => {
  const handleNameChange = (e: React.ChangeEvent<HTMLInputElement>) => {
    onChange({
      ...category,
      name: e.target.value,
      slug: e.target.value.toLowerCase().replace(/\s+/g, '-').replace(/[^a-z0-9-]/g, '')
    });
  };

  const handleOrderChange = (e: React.ChangeEvent<HTMLInputElement>) => {
    onChange({
      ...category,
      order: parseInt(e.target.value) || 0
    });
  };

  return (
    <div className="flex items-center space-x-2 mb-2">
      <Input
        className="flex-1"
        value={category.name}
        onChange={handleNameChange}
        placeholder="Kategori Adı"
      />
      <Input
        className="w-20"
        type="number"
        value={category.order}
        onChange={handleOrderChange}
        placeholder="Sıra"
      />
      <Button
        variant="ghost"
        size="icon"
        onClick={onDelete}
        className="text-destructive hover:text-destructive hover:bg-red-50"
      >
        <X className="h-4 w-4" />
      </Button>
    </div>
  );
};

export default function BulkCategoryImporter() {
  const { toast } = useToast();
  const { data: existingCategories, isLoading: isLoadingCategories, refetch } = useCategories();
  const [categories, setCategories] = useState<{name: string; slug: string; order: number}[]>([]);
  const [importing, setImporting] = useState(false);

  // Create category mutation
  const createCategoryMutation = useMutation({
    mutationFn: async (categoryData: {name: string; slug: string; order: number}) => {
      try {
        const res = await fetch("/api/categories", {
          method: "POST",
          headers: {
            "Content-Type": "application/json"
          },
          body: JSON.stringify(categoryData),
          credentials: "include"
        });
        
        if (!res.ok) {
          throw new Error(`Failed to create category: ${res.status}`);
        }
        
        return await res.json();
      } catch (error) {
        console.error("Category creation error:", error);
        throw error;
      }
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/categories"] });
    },
    onError: (error: any) => {
      toast({
        title: "Kategori eklenemedi",
        description: error.message || "Kategori eklenirken bir hata oluştu.",
        variant: "destructive"
      });
    }
  });

  const loadJouskaCategories = () => {
    setCategories(jouskaCategoryData);
    toast({
      title: "Jouska Kategorileri Yüklendi",
      description: "Jouska kategorileri başarıyla yüklendi. Kaydetmek için 'Kategorileri Kaydet' butonuna tıklayın.",
    });
  };

  const addNewCategory = () => {
    setCategories([...categories, {
      name: "",
      slug: "",
      order: categories.length > 0 ? Math.max(...categories.map(c => c.order)) + 1 : 1
    }]);
  };

  const updateCategory = (index: number, updatedCategory: {name: string; slug: string; order: number}) => {
    const newCategories = [...categories];
    newCategories[index] = updatedCategory;
    setCategories(newCategories);
  };

  const deleteCategory = (index: number) => {
    setCategories(categories.filter((_, i) => i !== index));
  };

  const saveCategories = async () => {
    if (categories.length === 0) {
      toast({
        title: "Kategori bulunamadı",
        description: "Kaydedilecek kategori bulunamadı.",
        variant: "destructive"
      });
      return;
    }

    // Validate categories
    const invalidCategories = categories.filter(c => !c.name.trim());
    if (invalidCategories.length > 0) {
      toast({
        title: "Geçersiz kategori(ler)",
        description: "Tüm kategorilerin adları doldurulmalıdır.",
        variant: "destructive"
      });
      return;
    }

    setImporting(true);
    try {
      // Save each category
      for (const category of categories) {
        // Check if category already exists by name
        const existingCategory = existingCategories?.find(c => 
          c.name.toLowerCase() === category.name.toLowerCase() ||
          c.slug === category.slug
        );

        // Skip if exists
        if (existingCategory) {
          console.log(`Category "${category.name}" already exists, skipping.`);
          continue;
        }

        // Create new category
        await createCategoryMutation.mutateAsync(category);
      }

      toast({
        title: "Kategoriler kaydedildi",
        description: "Kategoriler başarıyla kaydedildi.",
      });

      // Refresh categories
      await refetch();
    } catch (error) {
      console.error("Error saving categories:", error);
      toast({
        title: "Kategoriler kaydedilemedi",
        description: "Kategorileri kaydederken bir hata oluştu.",
        variant: "destructive"
      });
    } finally {
      setImporting(false);
    }
  };

  const importFromText = (e: React.ChangeEvent<HTMLTextAreaElement>) => {
    const text = e.target.value;
    const lines = text.split('\n').filter(line => line.trim() !== '');
    
    const newCategories = lines.map((line, index) => {
      return {
        name: line.trim(),
        slug: line.trim().toLowerCase().replace(/\s+/g, '-').replace(/[^a-z0-9-]/g, ''),
        order: index + 1
      };
    });
    
    setCategories(newCategories);
  };

  return (
    <Card className="w-full">
      <CardHeader>
        <CardTitle>Kategori Yönetimi</CardTitle>
        <CardDescription>
          Kategorileri toplu olarak içe aktarın veya düzenleyin.
        </CardDescription>
      </CardHeader>
      <CardContent>
        <div className="space-y-4">
          <div className="flex space-x-2">
            <Button 
              onClick={loadJouskaCategories} 
              variant="outline"
              className="flex-1"
            >
              Jouska Kategorilerini Yükle
            </Button>
            <Button 
              onClick={addNewCategory}
              variant="outline"
              className="flex items-center"
            >
              <Plus className="h-4 w-4 mr-2" />
              Yeni Kategori
            </Button>
          </div>

          <Separator className="my-4" />

          <div className="space-y-2">
            <Label htmlFor="multiline-import">Metin Olarak İçe Aktar</Label>
            <Textarea 
              id="multiline-import" 
              placeholder="Her satıra bir kategori yazın..." 
              rows={4}
              onChange={importFromText}
            />
            <p className="text-xs text-muted-foreground">
              Her bir satıra bir kategori adı yazın. Kategoriler otomatik olarak sıralanacaktır.
            </p>
          </div>

          <Separator className="my-4" />

          <div className="space-y-2">
            <div className="flex justify-between items-center">
              <Label>Kategoriler ({categories.length})</Label>
              <div className="text-xs text-muted-foreground">
                {existingCategories?.length || 0} mevcut kategori
              </div>
            </div>
            
            <div className="max-h-[300px] overflow-y-auto p-2 border rounded-md">
              {categories.length > 0 ? (
                categories.map((category, index) => (
                  <CategoryEditor 
                    key={index}
                    category={category}
                    onChange={(updatedCategory) => updateCategory(index, updatedCategory)}
                    onDelete={() => deleteCategory(index)}
                  />
                ))
              ) : (
                <div className="p-4 text-center text-muted-foreground">
                  Henüz kategori eklenmedi. Yeni kategori ekleyin veya var olan kategorileri yükleyin.
                </div>
              )}
            </div>
          </div>
        </div>
      </CardContent>
      <CardFooter>
        <Button 
          onClick={saveCategories} 
          disabled={importing || categories.length === 0}
          className="w-full"
        >
          {importing ? (
            <>
              <Loader2 className="h-4 w-4 mr-2 animate-spin" />
              Kategoriler Kaydediliyor...
            </>
          ) : (
            <>
              <Save className="h-4 w-4 mr-2" />
              Kategorileri Kaydet
            </>
          )}
        </Button>
      </CardFooter>
    </Card>
  );
}